///////////////////////////////////////////////////////////////////////////////
// Common stream interface that supports both 2D and 3D stream interfaces
///////////////////////////////////////////////////////////////////////////////
#include "Audio/AudioPCH.h"

#include "container/fpool.h"

// Our pool
FixedPool<CSIHandle, 32> s_csiPool(
#ifdef _DEBUG
"CSIHandle",
#endif
false);

DoublyLinkedList<CommonStreamInterface> CommonStreamInterface::s_autoDelete(true);

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
CSIHandle::CSIHandle(
CommonStreamInterface *stream) :
m_stream(stream)
{
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
CSIHandle::~CSIHandle()
{
	if (m_stream)
		m_stream->m_handle = NULL;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
CommonStreamInterface *CSIHandle::GetStream(CSIHandle *&handle)
{
	// Do we have a handle?
	if (!handle)
		return NULL;

	// Do we have a stream?
	CommonStreamInterface *stream = handle->GetStream();
	if (stream)
		return stream;

	// We lost our stream...clean up for them
	delete handle;
	handle = NULL;
	return NULL;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void CSIHandle::Kill(CSIHandle *&handle, bool stopPlaying)
{
	if (handle)
	{
		handle->Kill(stopPlaying);
		delete handle;
		handle = NULL;
	}
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void CSIHandle::Kill(bool stopPlaying)
{
	if (m_stream)
	{
		if (!stopPlaying)
			m_stream->Detach();
		else
		{
			delete m_stream;
			ASSERT(m_stream == NULL);
		}
	}
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void *CSIHandle::operator new(size_t size)
{
	// Sanity check
	ASSERT(size == sizeof(CSIHandle));

	// Chug one off the pool (should never fail)
	void *buffer = s_csiPool.NewPtr();
	ASSERT(buffer);

	// Hand it back to the system
	return buffer;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void CSIHandle::operator delete(void *me)
{
	// Do we have to check this?
	if (me)
	{
		CSIHandle *handle = (CSIHandle *)me;
	 	s_csiPool.Delete(handle);
	}
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
CommonStreamInterface::~CommonStreamInterface()
{
	// No more stream
	delete m_stream2D;
	delete m_stream3D;

	// We are dead!
	if (m_handle)
		m_handle->m_stream = NULL;

	// If we were in the auto-delete list free us
	if (m_autoDelete)
		s_autoDelete.Unlink(this);
}

///////////////////////////////////////////////////////////////////////////////
// Wee!
///////////////////////////////////////////////////////////////////////////////
CommonStreamInterface *CommonStreamInterface::Create2D(
SoundDriver &driver,
const char *fileName,
bool repeat,
bool autoDelete,
bool preLoad)
{
	SoundStream *stream = driver.CreateStream(fileName, repeat, preLoad);
	if (!stream)
		return NULL;

	CommonStreamInterface *csi = new CommonStreamInterface(*stream, autoDelete);
	if (csi && autoDelete)
		s_autoDelete.AddTail(csi);
	return csi;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
CommonStreamInterface *CommonStreamInterface::Create3D(
SoundDriver &driver,
const char *fileName,
bool repeat,
bool autoDelete,
SoundStream3D::Priority priority)
{
	SoundStream3D *stream = driver.CreateStream3D(fileName, repeat, priority);
	if (!stream)
		return NULL;

	CommonStreamInterface *csi = new CommonStreamInterface(*stream, autoDelete);
	if (csi && autoDelete)
		s_autoDelete.AddTail(csi);
	return csi;
}

///////////////////////////////////////////////////////////////////////////////
// Create using a song id
///////////////////////////////////////////////////////////////////////////////
CommonStreamInterface *CommonStreamInterface::Create2D(
SoundDriver &driver,
SoundDriver::SONGID id,
bool autoDelete,
bool repeat)
{
	SoundStream *stream = driver.CreateStream(id, repeat);
	if (!stream)
		return NULL;

	CommonStreamInterface *csi = new CommonStreamInterface(*stream, autoDelete);
	if (csi && autoDelete)
		s_autoDelete.AddTail(csi);
	return csi;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void CommonStreamInterface::UpdateAutoDelete()
{
	CommonStreamInterface *stream = s_autoDelete.Head();
	while (stream)
	{
		CommonStreamInterface *next = stream->next;

		// Delete?
		if (stream->m_autoDelete && !stream->IsPlaying())
		{
			stream->m_autoDelete = false;
			s_autoDelete.Remove(stream);
		}

		// Next!
		stream = next;
	}
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
int CommonStreamInterface::FlushAutoDelete()
{
	int flushed = 0;
	CommonStreamInterface *stream = s_autoDelete.Head();
	while (stream)
	{
		CommonStreamInterface *next = stream->next;

		// Delete?
		if (stream->IsPlaying())
			stream->Stop();
		stream->m_autoDelete = false;
		s_autoDelete.Remove(stream);
		++flushed;

		// Next!
		stream = next;
	}

	return flushed;
}

///////////////////////////////////////////////////////////////////////////////
// Create a handle for this stream (you must delete it!)
// Will assert if you already have it
///////////////////////////////////////////////////////////////////////////////
CSIHandle *CommonStreamInterface::CreateHandle()
{
	ASSERT(!m_handle);
	if (m_handle)
		return NULL;

	// Create it
	m_handle = new CSIHandle(this);
	return m_handle;
}

///////////////////////////////////////////////////////////////////////////////
// What is our name?
///////////////////////////////////////////////////////////////////////////////
const char *CommonStreamInterface::Name()
{
	if (m_stream2D)
		return m_stream2D->Name();
	else
		return m_stream3D->Name();
}

///////////////////////////////////////////////////////////////////////////////
// Repeat the buffer?
///////////////////////////////////////////////////////////////////////////////
bool CommonStreamInterface::Looping()
{
	if (m_stream2D)
		return m_stream2D->Looping();
	else
		return m_stream3D->Looping();
}

///////////////////////////////////////////////////////////////////////////////
// Manage the fader
///////////////////////////////////////////////////////////////////////////////
void CommonStreamInterface::BindFader(Fader *fader)
{
	if (m_stream2D)
		m_stream2D->BindFader(fader);
	else
		m_stream3D->BindFader(fader);
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void CommonStreamInterface::ClearFader()
{
	if (m_stream2D)
		m_stream2D->ClearFader();
	else
		m_stream3D->ClearFader();
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
Fader *CommonStreamInterface::GetFader()
{
	if (m_stream2D)
		return m_stream2D->GetFader();
	else
		return m_stream3D->GetFader();
}

///////////////////////////////////////////////////////////////////////////////
// Set the volume [0..1], 0.0 is silent 1.0 is full volume
///////////////////////////////////////////////////////////////////////////////
bool CommonStreamInterface::SetVolume(float volume)
{
	if (m_stream2D)
		return m_stream2D->SetVolume(volume);
	else
		return m_stream3D->SetVolume(volume);
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
float CommonStreamInterface::GetVolume()
{
	if (m_stream2D)
		return m_stream2D->GetVolume();
	else
		return m_stream3D->GetVolume();
}

///////////////////////////////////////////////////////////////////////////////
// Set the pan [-1..1] (stereo streams can't be panned)
///////////////////////////////////////////////////////////////////////////////
bool CommonStreamInterface::SetPan(float pan)
{
	if (m_stream2D)
		return m_stream2D->SetPan(pan);
	else
		return false;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
float CommonStreamInterface::GetPan()
{
	if (m_stream2D)
		return m_stream2D->GetPan();
	else
		return 0.0f;
}

///////////////////////////////////////////////////////////////////////////////
// Change the pitch of the sound in semitones
///////////////////////////////////////////////////////////////////////////////
bool CommonStreamInterface::SetPitch(float pitchShift)
{
	if (m_stream2D)
		return m_stream2D->SetPitch(pitchShift);
	else
		return m_stream3D->SetPitch(pitchShift);
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
float CommonStreamInterface::GetPitch()
{
	if (m_stream2D)
		return m_stream2D->GetPitch();
	else
		return m_stream3D->GetPitch();
}

///////////////////////////////////////////////////////////////////////////////
// Set the mix for this sound stream
///////////////////////////////////////////////////////////////////////////////
bool CommonStreamInterface::SetWetDry(float wet, float dry)
{
	if (m_stream2D)
		return m_stream2D->SetWetDry(wet, dry);
	else
		return m_stream3D->SetWetDry(wet, dry);
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
float CommonStreamInterface::GetWet()
{
	if (m_stream2D)
		return m_stream2D->GetWet();
	else
		return m_stream3D->GetWet();
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
float CommonStreamInterface::GetDry()
{
	if (m_stream2D)
		return m_stream2D->GetDry();
	else
		return m_stream3D->GetDry();
}

///////////////////////////////////////////////////////////////////////////////
// Playback control
///////////////////////////////////////////////////////////////////////////////
bool CommonStreamInterface::Play()
{
	if (m_stream2D)
		return m_stream2D->Play();
	else
		return m_stream3D->Play();
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
bool CommonStreamInterface::Stop(bool rewind)
{
	if (m_stream2D)
		return m_stream2D->Stop(rewind);
	else
		return m_stream3D->Stop(rewind);
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
bool CommonStreamInterface::IsPlaying()
{
	if (m_stream2D)
		return m_stream2D->IsPlaying();
	else
		return m_stream3D->IsPlaying();
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
int CommonStreamInterface::CurrentPlayPos()
{
	if (m_stream2D)
		return m_stream2D->CurrentPlayPos();
	else
		return m_stream3D->CurrentPlayPos();
}

///////////////////////////////////////////////////////////////////////////////
// Pause
///////////////////////////////////////////////////////////////////////////////
bool CommonStreamInterface::Pause()
{
	if (m_stream2D)
		return m_stream2D->Pause();
	else
		return m_stream3D->Pause();
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
bool CommonStreamInterface::Unpause()
{
	if (m_stream2D)
		return m_stream2D->Unpause();
	else
		return m_stream3D->Unpause();
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
bool CommonStreamInterface::Paused()
{
	if (m_stream2D)
		return m_stream2D->Paused();
	else
		return m_stream3D->Paused();
}

///////////////////////////////////////////////////////////////////////////////
// Detach the stream (make it auto cleanup)
///////////////////////////////////////////////////////////////////////////////
bool CommonStreamInterface::Detach()
{
	if (m_stream3D)
		m_stream3D->Detach();

	// Are we are already taken care of?
	if (!m_autoDelete)
	{
		m_autoDelete = true;
		s_autoDelete.AddTail(this);
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
t_Error CommonStreamInterface::StreamError()
{
	if (m_stream2D)
		return m_stream2D->StreamError();
	else
		return m_stream3D->StreamError();
}

///////////////////////////////////////////////////////////////////////////////
// Only the driver should create us. We own this stream
// and will call delete on it when done
///////////////////////////////////////////////////////////////////////////////
CommonStreamInterface::CommonStreamInterface(
SoundStream &stream,
bool autoDelete) :
m_autoDelete(autoDelete)
{
	prev = next = NULL;
	m_stream2D = &stream;
	m_stream3D = NULL;
	m_handle = NULL;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
CommonStreamInterface::CommonStreamInterface(
SoundStream3D &stream,
bool autoDelete) :
m_autoDelete(autoDelete)
{
	prev = next = NULL;
	m_stream2D = NULL;
	m_stream3D = &stream;
	m_handle = NULL;
}

