///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
#include "Audio/AudioPCH.h"

#include "Math/sfilter.h"
#include "Audio/fader.h"

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
Fader::Fader()
{
	m_name[0] = '\0';
	m_parent = NULL;
	m_volume = 0.0f;
	m_targetVolume = 0.0f;
	m_volumeChangeRate = 0.0f;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
Fader::Fader(
const char *name,
float volume,
Fader *parent) :
m_volume(volume),
m_parent(parent)
{
	ASSERT(strlen(name) < sizeof(m_name));
	strcpy(m_name, name);

	m_targetVolume = m_volume;
	m_volumeChangeRate = 0.0f;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void Fader::Setup(
const char *name,
float volume,
Fader *parent)
{
	ASSERT(strlen(name) < sizeof(m_name));
	strcpy(m_name, name);

	m_parent = parent;

	m_volume = volume;
	m_targetVolume = m_volume;
	m_volumeChangeRate = 0.0f;

	m_factor = 1.0f;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void Fader::SetVolume(
float targetVolume,
float time)
{
	m_targetVolume = Math::Clamp(targetVolume);

	if (time == 0.0f)
		m_volumeChangeRate = 0.0f;
	else
		m_volumeChangeRate = 1.0f / time;

	Update(0.0f);
}

///////////////////////////////////////////////////////////////////////////////
// Evaluate the current value
///////////////////////////////////////////////////////////////////////////////
float Fader::MixVolume()
{
	if (m_parent)
		return m_parent->MixVolume() * m_volume * m_factor;
	else
		return m_volume * m_factor;
}

///////////////////////////////////////////////////////////////////////////////
// Update (call this once per frame)
///////////////////////////////////////////////////////////////////////////////
void Fader::Update(
float timeElapsed)
{
	// approach the desired volume
	if (!Math::Zero(m_volumeChangeRate))
		Math::MoveToValue(&m_volume, m_targetVolume, timeElapsed * m_volumeChangeRate);
	else
		m_volume = m_targetVolume;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
FaderBoard::FaderBoard() :
m_faders(true)
{
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
FaderBoard::~FaderBoard()
{
}

///////////////////////////////////////////////////////////////////////////////
// Add a fader (I own it now, and will call delete)
///////////////////////////////////////////////////////////////////////////////
void FaderBoard::Add(
Fader *fader)
{
	if (fader)
		m_faders.AddTail(fader);
}

///////////////////////////////////////////////////////////////////////////////
// Remove a fader
///////////////////////////////////////////////////////////////////////////////
void FaderBoard::Delete(
Fader *&fader)
{
	if (fader)
	{
		m_faders.Remove(fader);
		fader = NULL;
	}
}

///////////////////////////////////////////////////////////////////////////////
// Remove a fader
///////////////////////////////////////////////////////////////////////////////
void FaderBoard::Remove(
Fader *fader)
{
	if (fader)
		m_faders.Unlink(fader);
}

///////////////////////////////////////////////////////////////////////////////
// Find a fader based on name
///////////////////////////////////////////////////////////////////////////////
Fader *FaderBoard::Find(
const char *name)
{
	Fader *fader = m_faders.Head();
	while (fader)
	{
		if (strcmp(fader->Name(), name) == 0)
			return fader;

		fader = fader->next;
	}

	return NULL;
}

///////////////////////////////////////////////////////////////////////////////
// Update all the faders
///////////////////////////////////////////////////////////////////////////////
void FaderBoard::Update(
float timeElapsed)
{
	Fader *fader = m_faders.Head();
	while (fader)
	{
		fader->Update(timeElapsed);
		fader = fader->next;
	}
}

