/*************************************/
/*                                   */
/*   SwooshMgr.h                     */
/*   big juju   10/18/02             */
/*   ned martin  avalanche software  */
/*   visual c++ 6.0                  */
/*   swooshes are dynamic trails     */
/*                                   */
/*************************************/

/* swooshes are dynamic trails that are generated by something that moves, sort of like a ribbon stretching out behind
a moving object. swooshes are fairly expensive to set up, so typically things that will need a swoosh create it at
level initialization time and activate it when needed */

/******************* includes ****************************/

#include "Game/GamePCH.h"



#include "container/ManagedList.h"
#include "EngineHelper/timer.h"
#include "Display/Common/StripBuffer.h"
#include "Display/Common/VtxBuffer.h"
#include "mathUtil/Ballistic.h"
#include "camera/director.h"
#include "Math/fastrand.h"

/******************* defines *****************************/

/* flags for ts_Swoosh.u16Flags-- first few bits are a type field */

#define SWOOSH_TYPE_BONE						0x0000
#define SWOOSH_TYPE_ENDPOINT					0x0001
#define SWOOSH_TYPE_BALLISTIC					0x0002
#define SWOOSH_TYPE_MASK						0x0003

/* flag bits start after type mask */

#define SWOOSH_ACTIVE							0x0004
#define SWOOSH_VISIBLE							0x0008
#define SWOOSH_ADD_NEW_POINT					0x0010	// may want to let old swoosh die off gracefully
#define SWOOSH_BONE_SWAP						0x0020	// set if fBoneStart = 1.0f and fBoneEnd = 0.0f-- swap order of endpoints
#define SWOOSH_BONE_ADJUST						0x0040	// set if fBoneStart and fBoneEnd are not default values
#define SWOOSH_LOD_SUSPENDED					0x0080	// beyond LOD distance from all active cameras
#define SWOOSH_NO_INTERMEDIATE_POINTS		0x0100	// swoosh source speed is predictable, don't need to check for intermediate points
#define SWOOSH_BOUNDING_BOX_VALID			0x0200	// swoosh's bounding box was recalculated
#define SWOOSH_NO_SHARP_TAIL					0x0400	// don't remove oldest triangle to give swoosh a sharp tail
#define SWOOSH_IS_BLUR							0x0800	// I can hack with the best of 'em

/* in meters. within this distance, no LOD degradation of swoosh */

#define HIRES_LOD_RANGE							5.0f

/* support putting a limit on how many swoosh groups will be created for a particular emitter-- those swoosh groups */
/* will be shared among all emitters of that name, on the assumption that only a few will be visible at any given time */

#define MAX_SHARED_SWOOSH_GROUP_EMITTERS	10

/******************* macros ******************************/

#define SWOOSH_GET_RECORD(Handle)		&SwooshMgr.pSwooshAlloc[HANDLE_INDEX(Handle)]

/******************* structures **************************/

/* swooshes work by tracking a series of points over time. each point is in world space, and is at the center */
/* of a line of some size and orientation. the swoosh is created by connecting the lines together to create a */
/* ribbon-looking kind of thing. typically, the lines get smaller as they age, so that the ribbon gets narrower */
/* towards the tail */

struct ts_SwooshPoint
{
	Vector3				vPoint;					// center point of line
	Vector3				vLineDir;				// vPoint +/- vLineDir are endpoints of line. this is a unit vector
	Vector3				vTangent;				// spline tangent pointing to previous point, for interpolation. unit vector
	float					fLineHalfLength;		// original magnitude of vLineDir before normalizing
	float					fLifetime;				// each point ages
};

/* an individual swoosh */

struct ts_Swoosh
{
	t_Handle				Handle;					//	assigned handle for access, invalid handle checking
	ts_Swoosh			*pNext;					// linked list pointer-- does not have to be first
	u16					u16Flags;
	int					nDataIndex;				// which of possibly-multiply-loaded particle data files swoosh comes from

	/* control method 1-- follow a bone-- only one control method can be active */

	CActor				*pCActor;				// optional bone that swoosh may be attached to
	int					nBoneID;
	float					fBoneStart;				// typically 0.0 to 1.0, where along bone endpoint 0 is, at 0.0 use start of bone
	float					fBoneEnd;				// typically 0.0 to 1.0, where along bone endpoint 1 is, at 0.0 use end of bone

	/* control method 2-- follow two endpoints-- only one control method can be active */

	Vector3Packed				*pvEndpoints[2];		// optional points (world coords) that swoosh may be attached to

	/* control method 3-- follow a ballistic path */

	CBallistic			*pBallisticPath;
	float					fPathHalfWidth;					// meters
	float					fPointInterval;					// meters

	/* each swoosh needs a history buffer */

	ts_SwooshPoint		*pSwooshPoints;
	int					nSwooshPointCount;
	int					nSwooshPointIndex;	// the buffer is circular

	/* the lifetime of a new point in the history buffer controls how long the swoosh persists */

	float					fPointLifetime;

	/* the geometry implementation */

	StripBuffer			*pStripBuffer;
	VtxBuffer			*pVtxBuffer;

	/* world-space bounding box, recalculated each time vertex buffer is rebuilt */

	ts_BoundingBox		WorldBoundingBox;

	/* source data for this swoosh-- includes envelopes.very simple swooshes can be created without source data */

	ts_SwooshSource	*pSwooshSource;

	/* for very simple swooshes that are created with pigpen-designed source. only valid if pSwooshSource is NULL */

	ts_bRGBA				Color;
	float					fMaxAlpha;				// in 0..255 range, but float

	/* edge 2 multipliers-- basically to allow fading across swoosh, as opposed to over time */

	float					fEdge2ColorMult;
	float					fEdge2AlphaMult;

	/* can be used for tying swoosh to collision color data */

	float					*pfColorMult;			// 0.0 to 1.0
	float					fColorMultFloor;		// 0.0 to 1.0

	/* can be used to tie a swooshes constant alpha to its parent's constant alpha */

	float					*pfConstantAlpha;

	/* LOD control. beyond LOD distance swoosh isn't active. as swoosh approaches LOD distance it fades out, and uses */
	/* fewer (if any) intermediate points */

	float					fLODDistance;
	float					fDistanceToNearestCamera;
};

/* swoosh manager */

struct ts_SwooshMgr
{
	ts_SLinkList		SwooshFree;				// linked list of free records
	ts_SLinkList		SwooshActive;			// linked list of active records
	ts_Swoosh			*pSwooshAlloc;			// pointer to malloc'd records
	u16					u16ID;					// for assigning handles
	int					nMaxSwooshes;			// record passed value for debugging, not really necessary otherwise
	int					nMaxSwooshEmitters;	// record passed value for debugging, not really necessary otherwise

	/* a managed list of CSwooshEmitter records. this provides a pool of records to allocate from and a linked list */
	/* of active records. we allocate this at the max size needed and lock it-- it never grows or shrinks. it could */
	/* certainly grow if we wanted it to-- that's the way the managed list is set up-- but i still prefer to malloc */
	/* all we need and then find when we need more rather than having a bunch of lists that can all grow and present */
	/* unpredictable memory issues later on (e.g. expand to take up all memory so nothing's left for transient */
	/* malloc's). as its stands, the swoosh manager is initialized once at game init time, so the max has to */
	/* satisfy all game levels. i'd like to see this move to per-level max values */

	ManagedList<CSwooshEmitter>	*d_EmitterList;	// won't be a pointer when SwooshMgr is a class...

	/* swooshes are memory-handle intensive, and each emitter may need a lot of swooshes. to reduce memory */
	/* use, the caller can set limits on how many emitters of a particular name will actually create a group */
	/* of swooshes. if an emitter is active but does not have any swooshes, it looks for inactive emitter of */
	/* the same type to steal swooshes from */

	ts_SwooshEmitterSource			*d_pSharedSwooshGroupEmitterSource[MAX_SHARED_SWOOSH_GROUP_EMITTERS];
	int									d_nMaxSharedSwooshGroups[MAX_SHARED_SWOOSH_GROUP_EMITTERS];
	int									d_nSharedSwooshGroups[MAX_SHARED_SWOOSH_GROUP_EMITTERS];
	int									d_nDeactivatedID;	// so i can tell which emitter was deactivated longest ago

};

/******************* externs *****************************/

/******************* global variables ********************/

/******************* local variables *********************/

/* swoosh manager */

static ts_SwooshMgr SwooshMgr;

static FastRandom SwooshRandom(0);	// swoosh manager has its own random number sequence

/******************* local prototypes ********************/

ts_Swoosh *SwooshMgr__AllocSwoosh();
t_Handle SwooshMgr__CreateSwooshFromRecord(
						int nDataIndex,
						int nSwooshID,
						Vector3Packed *pEndpoint0,
						Vector3Packed *pEndpoint1,
						float fPointCountMultiplier);
ts_Swoosh *Swoosh__Create(
						ts_SwooshSource *pSwooshSource,
						int nDataIndex,
						float fPointCountMultiplier,
						int nForcePointCount,			// for ballistic swoosh support, 0 means ignore
						/* the following arguments are for swooshes without source data-- created directly */
						ts_bRGBA *pColor = NULL,
						float fMaxAlpha = 0.0f,
						float fPointLifetime = 0.0f,
						bool bAdditiveRender = false,
						float fLODDistance = 0.0f);
inline Vector2 Swoosh__WorldToUV(const Matrix4x4 &worldToScreen, const Vector3 &world, float dwell = 0.01f);
void Swoosh__Advance(const Matrix4x4 &worldToScreen, ts_Swoosh *pSwoosh, float fFrameSeconds);
void Swoosh__AddPoint(ts_Swoosh *pSwoosh, float fFrameSeconds);
void Swoosh__AddPoint(ts_Swoosh *pSwoosh, float fFrameSeconds, Vector3 *vEndpoints);
void Swoosh__Interpolate(ts_Swoosh *pSwoosh, ts_SwooshPoint *pPreviousSwooshPoint, ts_SwooshPoint *pNewSwooshPoint);
void Swoosh__CalculateBallisticPath(ts_Swoosh *pSwoosh, float fFrameSeconds);
void Swoosh__Kill(ts_Swoosh *pSwoosh);
float Swoosh__EvaluateEnvelope(ts_SwooshSource *pSwooshSource, int nIndex, float fT);
Vector3 SwooshPoint__CalculateTangent(ts_SwooshPoint *pPoint1, ts_SwooshPoint *pPoint2);
void Swoosh__InterpolateColor(ts_SwooshSource *pSwooshSource, float fValue, ts_bRGBA &Color);
//yuch-- delete this, maybe make it a public static function of particle mgr when particle mgr is a c++ class?
float SwooshMgr__fRandomValue(float fValue, int nRandomPctValue);

/******************* functions ***************************/

/* initialize swoosh system */

int SwooshMgr__Init(int nMaxSwooshes, int nMaxSwooshEmitters)
{
	/* clear the manager */

	memset(&SwooshMgr, 0, sizeof(ts_SwooshMgr));

	/* done if no swooshes */

	if (nMaxSwooshes == 0)
		return(TRUE);
	SwooshMgr.nMaxSwooshes = nMaxSwooshes;
	SwooshMgr.nMaxSwooshEmitters = nMaxSwooshEmitters;

	/* allocate records, initialize linked lists */

	MEM_SET_ALLOC_NAME("SwooshMgr list");
	SwooshMgr.pSwooshAlloc = (ts_Swoosh *) memAlloc(SwooshMgr.nMaxSwooshes * sizeof(ts_Swoosh));
	SLinkList__Init(&SwooshMgr.SwooshFree, offsetof(ts_Swoosh, pNext));
	SLinkList__LinkBlock(&SwooshMgr.SwooshFree, SwooshMgr.pSwooshAlloc, SwooshMgr.nMaxSwooshes, sizeof(ts_Swoosh));
	SLinkList__Init(&SwooshMgr.SwooshActive, offsetof(ts_Swoosh, pNext));

	/* initialize records */

	ts_Swoosh *pSwoosh = SwooshMgr.pSwooshAlloc;
	for (int i = 0; i < SwooshMgr.nMaxSwooshes; i++, pSwoosh++)
	{
		pSwoosh->Handle = INVALID_HANDLE;
		pSwoosh->u16Flags = 0;
	}

	/* swoosh emitter list */

	SwooshMgr.d_EmitterList = new ManagedList<CSwooshEmitter>(nMaxSwooshEmitters, 0);	// does not call constructor on each record
	SwooshMgr.d_EmitterList->Lock();	// must malloc all that we'll ever need for memory leak purposes

	/* no shared-swoosh-group limits */

	memset(SwooshMgr.d_pSharedSwooshGroupEmitterSource, 0, MAX_SHARED_SWOOSH_GROUP_EMITTERS * sizeof(ts_SwooshEmitterSource *));

	/* success */


	return(TRUE);
}

/* reset */

void SwooshMgr__Reset(void)
{
	/* kill all swoosh emitters and swooshes */

	SwooshMgr__KillAllEmitters();	// do this first since emitter will kill its own swooshes
	SwooshMgr__KillAllSwooshes();

	/* no shared-swoosh-group limits */

	memset(SwooshMgr.d_pSharedSwooshGroupEmitterSource, 0, MAX_SHARED_SWOOSH_GROUP_EMITTERS * sizeof(ts_SwooshEmitterSource *));

	/* all swooshes should be in free list now, ready to go */

	ASSERT(SLinkList__Check(&SwooshMgr.SwooshFree) == (signed) SwooshMgr.nMaxSwooshes);

	/* clear all handles */

	ts_Swoosh *pSwoosh = (ts_Swoosh *) SwooshMgr.SwooshFree.pHead;
	BoundingBox__Clear(&pSwoosh->WorldBoundingBox);
	while (pSwoosh)
	{
		pSwoosh->Handle = INVALID_HANDLE;
		pSwoosh = pSwoosh->pNext;
	}
}

/* shutdown */

void SwooshMgr__Shutdown(void)
{
	/* kill all swoosh emitters and swooshes */

	SwooshMgr__KillAllEmitters();	// do this first since emitter will kill its own swooshes
	SwooshMgr__KillAllSwooshes();

	/* delete managed lists */

	delete SwooshMgr.d_EmitterList;
	SwooshMgr.d_EmitterList = NULL;

	/* shut down */

	if (SwooshMgr.pSwooshAlloc)
		memFree(SwooshMgr.pSwooshAlloc);
	memset(&SwooshMgr, 0, sizeof(ts_SwooshMgr));
}

ts_Swoosh *SwooshMgr__AllocSwoosh()
{
	if (SwooshMgr.SwooshFree.pHead == NULL)
		return(NULL);	// no free swoosh record
	ts_Swoosh *pSwoosh = (ts_Swoosh *) SLinkList__GetAndRemoveHead(&SwooshMgr.SwooshFree);
	/* build handle */

	ASSERT(pSwoosh->Handle == INVALID_HANDLE);			// record should not have a handle assigned
	HANDLE_INDEX(pSwoosh->Handle) = pSwoosh - SwooshMgr.pSwooshAlloc;	// index into allocated array of records
	HANDLE_ID(pSwoosh->Handle) = SwooshMgr.u16ID++;
	if (pSwoosh->Handle == INVALID_HANDLE)
		HANDLE_ID(pSwoosh->Handle) = SwooshMgr.u16ID++;	// do it again to avoid INVALID_HANDLE
	ASSERT(pSwoosh->Handle != INVALID_HANDLE);			// record should have a valid handle

	/* add to head of active list */

	SLinkList__AddLinkToHead(&SwooshMgr.SwooshActive, pSwoosh);

	/* clear record */

	pSwoosh->u16Flags = 0;
	pSwoosh->pVtxBuffer = NULL;
	pSwoosh->pStripBuffer = NULL;

	/* we don't know control method yet */

	pSwoosh->pCActor = NULL;
	pSwoosh->pvEndpoints[0] = NULL;
	pSwoosh->pvEndpoints[1] = NULL;
	pSwoosh->pBallisticPath = NULL;

	/* done */

	BoundingBox__Clear(&pSwoosh->WorldBoundingBox);

	return(pSwoosh);
}

void SwooshMgr__KillAllSwooshes(t_Handle Handle /*= INVALID_HANDLE*/)
{
	/* if handle specified, kill all records that match-- this is useful for unloading a particle database file */

	int nDataIndex = -1;
	if (Handle != INVALID_HANDLE)
	{
		nDataIndex = ParticleMgr__GetDataIndex(Handle);
		if (nDataIndex == MAX_PARTICLE_DATA_FILES)
			return;	// handle not found
	}

	/* loop through active records */

	ts_Swoosh *pNextSwoosh = (ts_Swoosh *) SwooshMgr.SwooshActive.pHead;
	while (pNextSwoosh)
	{
		/* pre-load next since this one may be deleted */

		ts_Swoosh *pSwoosh = pNextSwoosh;
		pNextSwoosh = pNextSwoosh->pNext;

		/* if no handle specified, always kill it, otherwise kill it if it belongs to this group */

		if ((Handle == INVALID_HANDLE) || (pSwoosh->nDataIndex == nDataIndex))
			Swoosh__Kill(pSwoosh);
	}

#ifdef _DEBUG
	/* if no handle specified, all records should be free now */

	if (Handle == INVALID_HANDLE)
		ASSERT(SLinkList__Check(&SwooshMgr.SwooshFree) == (signed) SwooshMgr.nMaxSwooshes);	// all swooshes should be free now
#endif //_DEBUG
}

void SwooshMgr__KillAllEmitters(t_Handle Handle /*= INVALID_HANDLE*/)
{
	/* if handle specified, kill all records that match-- this is useful for unloading a particle database file */

	int nDataIndex = -1;
	if (Handle != INVALID_HANDLE)
	{
		nDataIndex = ParticleMgr__GetDataIndex(Handle);
		if (nDataIndex == MAX_PARTICLE_DATA_FILES)
			return;	// handle not found
	}

	/* loop through active records */

	CSwooshEmitter *pNextSwooshEmitter = SwooshMgr.d_EmitterList->ActiveHead();
	while (pNextSwooshEmitter)
	{
		/* pre-load next since this one may be deleted */

		CSwooshEmitter *pSwooshEmitter = pNextSwooshEmitter;
		pNextSwooshEmitter = pNextSwooshEmitter->next;

		/* if no handle specified, always kill it, otherwise kill it if it belongs to this group */

		if ((Handle == INVALID_HANDLE) || (pSwooshEmitter->GetDataIndex() == nDataIndex))
			SwooshMgr.d_EmitterList->Delete(pSwooshEmitter);
	}

#ifdef _DEBUG
	/* if no handle specified, all records should be free now */

	if (Handle == INVALID_HANDLE)
		ASSERT(SwooshMgr.d_EmitterList->IsActiveEmpty());
#endif //_DEBUG
}

/* create a swoosh that is defined by the progress of two endpoints */

t_Handle SwooshMgr__CreateSwoosh(
				Vector3Packed *pEndpoint0,
				Vector3Packed *pEndpoint1,
				ts_bRGBA &Color,
				float fMaxAlpha,
				float fPointLifetime,
				bool bAdditiveRender,
				float fPointCountMultiplier /*= DEFAULT_POINT_COUNT_MULTIPLIER*/,
				float fLODDistance /*= SWOOSH_DEFAULT_LOD_DISTANCE*/)
{
	/* create simple swoosh-- no pigpen-created source data */

	ts_Swoosh *pSwoosh = Swoosh__Create(
									NULL,							// ts_SwooshSource *pSwooshSource
									-1,							// int nDataIndex
									fPointCountMultiplier,	// float fPointCountMultiplier
									0,								// int nForcePointCount
									&Color,						// ts_bRGBA *pColor = NULL
									fMaxAlpha,					// float fMaxAlpha = 0.0f
									fPointLifetime,			// float fPointLifetime = 0.0f
									bAdditiveRender,			// bool bAdditiveRender = 0.0f
									fLODDistance				// float fLODDistance = 0.0f
									);
	if (pSwoosh == NULL)
		return(INVALID_HANDLE);

	/* set the type */

	ASSERT((pSwoosh->u16Flags & SWOOSH_TYPE_MASK) == 0);
	pSwoosh->u16Flags |= SWOOSH_TYPE_ENDPOINT;

	/* set the endpoint pointers */

	pSwoosh->pvEndpoints[0] = pEndpoint0;
	pSwoosh->pvEndpoints[1] = pEndpoint1;
	ASSERT(pSwoosh->pCActor == NULL);	// swoosh can have only one control

	/* success */

	return(pSwoosh->Handle);
}

/* create a swoosh that is defined by the movement of a bone */

t_Handle SwooshMgr__CreateSwoosh(
				CActor *pCActor,
				int nBoneID,
				ts_bRGBA &Color,
				float fMaxAlpha,
				float fPointLifetime,
				bool bAdditiveRender,
				float fPointCountMultiplier /*= DEFAULT_POINT_COUNT_MULTIPLIER*/,
				float fLODDistance /*= SWOOSH_DEFAULT_LOD_DISTANCE*/,
				float fBoneStart /*= 0.0f*/,
				float fBoneEnd /*= 1.0f*/)
{
	/* create simple swoosh-- no pigpen-created source data */

	ts_Swoosh *pSwoosh = Swoosh__Create(
									NULL,							// ts_SwooshSource *pSwooshSource
									-1,							// int nDataIndex
									fPointCountMultiplier,	// float fPointCountMultiplier
									0,								// int nForcePointCount
									&Color,						// ts_bRGBA *pColor = NULL
									fMaxAlpha,					// float fMaxAlpha = 0.0f
									fPointLifetime,			// float fPointLifetime = 0.0f
									bAdditiveRender,			// bool bAdditiveRender = 0.0f
									fLODDistance				// float fLODDistance = 0.0f
									);
	if (pSwoosh == NULL)
		return(INVALID_HANDLE);

	/* set the type */

	ASSERT((pSwoosh->u16Flags & SWOOSH_TYPE_MASK) == 0);
	pSwoosh->u16Flags |= SWOOSH_TYPE_BONE;

	/* set the bone pointer-- same code as SwooshMgr__CreateSwooshFromRecord */

	pSwoosh->pCActor = pCActor;
	pSwoosh->nBoneID = nBoneID;
	pSwoosh->fBoneStart = fBoneStart;
	pSwoosh->fBoneEnd = fBoneEnd;
	if ((fBoneStart == 1.0f) && (fBoneEnd == 0.0f))
		pSwoosh->u16Flags |= SWOOSH_BONE_SWAP;		// swap endpoint order
	else if ((fBoneStart != 0.0f) || (fBoneEnd != 1.0f))
		pSwoosh->u16Flags |= SWOOSH_BONE_ADJUST;	// need more calculation
	ASSERT(pSwoosh->pvEndpoints[0] == NULL);		// swoosh can have only one control

	/* success */

	return(pSwoosh->Handle);
}

/* create a swoosh that is defined by a ballistic path-- entire history buffer gets recalculated every time */

t_Handle SwooshMgr__CreateSwoosh(
				CBallistic *pBallisticPath,
				float fPathWidth,							// meters
				float fPointInterval,					// seconds
				float fPathMaxTime,						// seconds
				ts_bRGBA &Color,
				float fMaxAlpha,
				float fPointLifetime,
				bool bAdditiveRender,
				float fLODDistance /*= SWOOSH_DEFAULT_LOD_DISTANCE*/)
{
	/* we know number of points needed to represent this path */

	int nForcePointCount = (int) (fPathMaxTime / fPointInterval) + 1;

	/* create simple swoosh-- no pigpen-created source data */

	ts_Swoosh *pSwoosh = Swoosh__Create(
									NULL,							// ts_SwooshSource *pSwooshSource
									-1,							// int nDataIndex
									1.0f,							// float fPointCountMultiplier
									nForcePointCount,			// int nForcePointCount
									&Color,						// ts_bRGBA *pColor = NULL
									fMaxAlpha,					// float fMaxAlpha = 0.0f
									fPointLifetime,			// float fPointLifetime = 0.0f
									bAdditiveRender,			// bool bAdditiveRender = 0.0f
									fLODDistance				// float fLODDistance = 0.0f
									);
	if (pSwoosh == NULL)
		return(INVALID_HANDLE);

	/* set the type */

	ASSERT((pSwoosh->u16Flags & SWOOSH_TYPE_MASK) == 0);
	pSwoosh->u16Flags |= SWOOSH_TYPE_BALLISTIC;

	/* no sharp tail for this kind of swoosh */

	pSwoosh->u16Flags |= SWOOSH_NO_SHARP_TAIL;

	/* record control data-- same code as SwooshMgr__CreateSwooshFromRecord */

	pSwoosh->pBallisticPath = pBallisticPath;
	pSwoosh->fPathHalfWidth = 0.5f * fPathWidth;
	pSwoosh->fPointInterval = fPointInterval;

	/* success */

	return(pSwoosh->Handle);
}

/* create a swoosh that is defined by the progress of two endpoints */

t_Handle SwooshMgr__CreateSwooshFromRecord(
				const char *pSwooshName,
				Vector3Packed *pEndpoint0,
				Vector3Packed *pEndpoint1,
				float fPointCountMultiplier /*= DEFAULT_POINT_COUNT_MULTIPLIER*/)
{
	/* find swoosh by name-- swoosh data is bundled in with particle data, so ask particle manager */

	int nDataIndex;
	int nSwooshIndex;
	if (ParticleMgr__FindSwooshByName(pSwooshName, nDataIndex, nSwooshIndex) == false)
		return(INVALID_HANDLE);	// not found

	/* create */

	return(SwooshMgr__CreateSwooshFromRecord(nDataIndex, nSwooshIndex, pEndpoint0, pEndpoint1, fPointCountMultiplier));
}

/* create a swoosh that is defined by the progress of two endpoints. this version is for internal use by swoosh emitters */

t_Handle SwooshMgr__CreateSwooshFromRecord(
						int nDataIndex,
						int nSwooshID,
						Vector3Packed *pEndpoint0,
						Vector3Packed *pEndpoint1,
						float fPointCountMultiplier)
{
	/* point to source for this swoosh */

	ts_SwooshSource *pSwooshSource = (ts_SwooshSource *) ParticleMgr__GetSwooshSource(nDataIndex, nSwooshID);

	/* create swoosh */

	ts_bRGBA White = {255, 255, 255, 255};	// not used, color comes from source envelope
	ts_Swoosh *pSwoosh = Swoosh__Create(
									pSwooshSource,				// ts_SwooshSource *pSwooshSource
									nDataIndex,					// int nDataIndex
									fPointCountMultiplier,	// float fPointCountMultiplier
									0								// int nForcePointCount
									// ts_bRGBA *pColor = NULL
									// float fMaxAlpha = 0.0f
									// float fPointLifetime = 0.0f
									// bool bAdditiveRender = 0.0f
									// float fLODDistance = 0.0f
									);
	if (pSwoosh == NULL)
		return(INVALID_HANDLE);

	/* set the type */

	ASSERT((pSwoosh->u16Flags & SWOOSH_TYPE_MASK) == 0);
	pSwoosh->u16Flags |= SWOOSH_TYPE_ENDPOINT;

	/* set the endpoint pointers */

	pSwoosh->pvEndpoints[0] = pEndpoint0;
	pSwoosh->pvEndpoints[1] = pEndpoint1;
	ASSERT(pSwoosh->pCActor == NULL);	// swoosh can have only one control

	/* success */

	return(pSwoosh->Handle);
}

/* create a swoosh from the particle data that is defined by the movement of a bone */

t_Handle SwooshMgr__CreateSwooshFromRecord(
				const char *pSwooshName,
				CActor *pCActor,
				int nBoneID,
				float fBoneStart /*= 0.0f*/,
				float fBoneEnd /*= 1.0f*/,
				float fPointCountMultiplier /*= DEFAULT_POINT_COUNT_MULTIPLIER*/)
{
	/* find swoosh by name-- swoosh data is bundled in with particle data, so ask particle manager */

	int nDataIndex;
	int nSwooshIndex;
	if (ParticleMgr__FindSwooshByName(pSwooshName, nDataIndex, nSwooshIndex) == false)
		return(INVALID_HANDLE);	// not found

	/* point to source for this swoosh */

	ts_SwooshSource *pSwooshSource = (ts_SwooshSource *) ParticleMgr__GetSwooshSource(nDataIndex, nSwooshIndex);

	/* create swoosh */

	ts_bRGBA White = {255, 255, 255, 255};	// not used, color comes from source envelope
	ts_Swoosh *pSwoosh = Swoosh__Create(
									pSwooshSource,				// ts_SwooshSource *pSwooshSource
									nDataIndex,					// int nDataIndex
									fPointCountMultiplier,	// float fPointCountMultiplier
									0								// int nForcePointCount
									// ts_bRGBA *pColor = NULL
									// float fMaxAlpha = 0.0f
									// float fPointLifetime = 0.0f
									// bool bAdditiveRender = 0.0f
									// float fLODDistance = 0.0f
									);
	if (pSwoosh == NULL)
		return(INVALID_HANDLE);

	/* set the type */

	ASSERT((pSwoosh->u16Flags & SWOOSH_TYPE_MASK) == 0);
	pSwoosh->u16Flags |= SWOOSH_TYPE_BONE;

	/* set the bone pointer-- same code as SwooshMgr__CreateSwoosh */

	pSwoosh->pCActor = pCActor;
	pSwoosh->nBoneID = nBoneID;
	pSwoosh->fBoneStart = fBoneStart;
	pSwoosh->fBoneEnd = fBoneEnd;
	if ((fBoneStart == 1.0f) && (fBoneEnd == 0.0f))
		pSwoosh->u16Flags |= SWOOSH_BONE_SWAP;		// swap endpoint order
	else if ((fBoneStart != 0.0f) || (fBoneEnd != 1.0f))
		pSwoosh->u16Flags |= SWOOSH_BONE_ADJUST;	// need more calculation
	ASSERT(pSwoosh->pvEndpoints[0] == NULL);		// swoosh can have only one control

	/* success */

	return(pSwoosh->Handle);
}

/* create a swoosh from the particle database that is defined by a ballistic path. this is a departure from typical
swooshes. typically, a swoosh has a history buffer of points. on point is added each frame. this gives the basic
swoosh effect-- a ribbon that follows some source. ballistic swooshes, on the other hand, have their history buffer
completely replaced each frame, with points that follow the ballistic path */

t_Handle SwooshMgr__CreateSwooshFromRecord(
				const char *pSwooshName,
				CBallistic *pBallisticPath,
				float fPathWidth,							// meters
				float fPointInterval,					// seconds
				float fPathMaxTime)						// seconds
{
	/* find swoosh by name-- swoosh data is bundled in with particle data, so ask particle manager */

	int nDataIndex;
	int nSwooshIndex;
	if (ParticleMgr__FindSwooshByName(pSwooshName, nDataIndex, nSwooshIndex) == false)
		return(INVALID_HANDLE);	// not found

	/* point to source for this swoosh */

	ts_SwooshSource *pSwooshSource = (ts_SwooshSource *) ParticleMgr__GetSwooshSource(nDataIndex, nSwooshIndex);

	/* we know number of points needed to represent this path */

	int nForcePointCount = (int) (fPathMaxTime / fPointInterval) + 1;

	/* create swoosh */

	ts_bRGBA White = {255, 255, 255, 255};	// not used, color comes from source envelope
	ts_Swoosh *pSwoosh = Swoosh__Create(
									pSwooshSource,				// ts_SwooshSource *pSwooshSource
									nDataIndex,					// int nDataIndex
									1.0,							// float fPointCountMultiplier
									nForcePointCount			// int nForcePointCount
									// ts_bRGBA *pColor = NULL
									// float fMaxAlpha = 0.0f
									// float fPointLifetime = 0.0f
									// bool bAdditiveRender = 0.0f
									// float fLODDistance = 0.0f
									);
	if (pSwoosh == NULL)
		return(INVALID_HANDLE);

	/* set the type */

	ASSERT((pSwoosh->u16Flags & SWOOSH_TYPE_MASK) == 0);
	pSwoosh->u16Flags |= SWOOSH_TYPE_BALLISTIC;

	/* no sharp tail for this kind of swoosh */

	pSwoosh->u16Flags |= SWOOSH_NO_SHARP_TAIL;

	/* record control data-- same code as SwooshMgr__CreateSwoosh */

	pSwoosh->pBallisticPath = pBallisticPath;
	pSwoosh->fPathHalfWidth = 0.5f * fPathWidth;
	pSwoosh->fPointInterval = fPointInterval;

	/* success */

	return(pSwoosh->Handle);
}

/* create a swoosh that is defined by the progress of two endpoints */

CSwooshEmitter *SwooshMgr__CreateEmitterFromRecord(const char *pName, Vector3 *pvPos, bool bStationary)
{
	/* find swoosh emitter by name-- swoosh data is bundled in with particle data, so ask particle manager */

	int nDataIndex;
	int nSwooshEmitterIndex;
	if (ParticleMgr__FindSwooshEmitterByName(pName, nDataIndex, nSwooshEmitterIndex) == false)
		return(NULL);	// not found

	/* get a new record from managed list */

	CSwooshEmitter *pSwooshEmitter = SwooshMgr.d_EmitterList->New();
	if (pSwooshEmitter == NULL)
	{
		ASSERT(false);	// allocate more swoosh emitters as part of swoosh manager initialization
		return(NULL);
	}

	/* point to source for this swoosh emitter */

	ts_SwooshEmitterSource *pSource = (ts_SwooshEmitterSource *) ParticleMgr__GetSwooshEmitterSource(nDataIndex, nSwooshEmitterIndex);

	/* get a free swoosh emitter, set common data */

	pSwooshEmitter->Initialize(pSource, nDataIndex, pvPos, bStationary);

	/* success */

	return(pSwooshEmitter);
}

/* release a emitter */

void SwooshMgr__KillEmitter(CSwooshEmitter *pSwooshEmitter)
{
	SwooshMgr.d_EmitterList->Delete(pSwooshEmitter);	// remove from active list, calls CSwooshEmitter destructor
}

/* set a limit on how many emiters of this name will create a swoosh group. all emitters of this name will share the
limited number of swoosh groups */

bool SwooshMgr__SetEmitterMaxSwooshGroups(const char *pName, int nMaxSwooshGroups)
{
	/* find swoosh emitter by name-- swoosh data is bundled in with particle data, so ask particle manager */

	int nDataIndex;
	int nSwooshEmitterIndex;
	if (ParticleMgr__FindSwooshEmitterByName(pName, nDataIndex, nSwooshEmitterIndex) == false)
		return(false);	// not found

	/* point to source for this swoosh emitter */

	ts_SwooshEmitterSource *pSource = (ts_SwooshEmitterSource *) ParticleMgr__GetSwooshEmitterSource(nDataIndex, nSwooshEmitterIndex);

	/* look for a slot in array that tracks registered emitters */

	int i;
	for (i = 0; i < MAX_SHARED_SWOOSH_GROUP_EMITTERS; i++)
	{
		if (SwooshMgr.d_pSharedSwooshGroupEmitterSource[i] == NULL)
			break;
	}
	if (i == MAX_SHARED_SWOOSH_GROUP_EMITTERS)
	{
		ASSERTS(false, "No slot available to register emitter for limited swoosh groups");
		return(false);	// no room
	}

	/*success-- record source, max */

	SwooshMgr.d_pSharedSwooshGroupEmitterSource[i] = pSource;
	SwooshMgr.d_nMaxSharedSwooshGroups[i] = nMaxSwooshGroups;
	SwooshMgr.d_nSharedSwooshGroups[i] = 0;
	return(true);
}

inline Vector2 Swoosh__WorldToUV(
const Matrix4x4 &worldToScreen,
const Vector3 &world,
float dwell)
{
	// Do the math
	Vector4 t = world.Transform3(worldToScreen);

	// Figure out where it lies on the screen
	float oow = 1.0f / t.w();
	Vector2 screen(((t.x() * oow) * 0.5f + 0.5f), ((t.y() * oow) * 0.5f + 0.5f));

	Vector2 tweak(SwooshRandom.InRange(-dwell, dwell), SwooshRandom.InRange(-dwell, dwell));
	screen += tweak;

	float uScale, vScale;
#if defined(PS2) || defined(_XBOX)
	TextureMgr::GetBackBufferTextureScaling( uScale, vScale );
#elif defined(DIRECTX_PC)
	//TextureMgr::GetBackBufferTextureScaling( s_uScale, s_vScale );
	uScale = RenderToTexture::renderSurface->GetImage()->GetImageUScale();
	vScale = RenderToTexture::renderSurface->GetImage()->GetImageVScale();
#else
	uScale = 1.0f;
	vScale = 1.0f;
#endif

	Vector2 uv;
	uv.x(Math::Clamp(screen.x() * uScale, 0.0f, uScale));
	uv.y(Math::Clamp(screen.y() * vScale, 0.0f, vScale));

	return uv;
}

/* update swoosh system-- record latest position of swoosh. this routine needs to be called after the actors are updated
(by g_stage.AdvanceSimulation), but before scene is rendered. it works to have this called by GameEngine__Advance, since
that is called in between the two */

void SwooshMgr__Advance(void)
{
	/* get time slice, skip advance if paused */

	float fFrameSeconds = g_timer.GetFrameSec();
	if (fFrameSeconds == 0.0f)
		return;

	/* loop through swoosh emitters */

	CSwooshEmitter *pNextSwooshEmitter = SwooshMgr.d_EmitterList->ActiveHead();
	while (pNextSwooshEmitter)
	{
		/* pre-load because we may delete this emitter */

		CSwooshEmitter *pSwooshEmitter = pNextSwooshEmitter;
		pNextSwooshEmitter = pNextSwooshEmitter->next;

		/* advance swoosh emitter, maybe it dies */

		if (pSwooshEmitter->Advance(fFrameSeconds) == false)
			SwooshMgr.d_EmitterList->Delete(pSwooshEmitter);	// remove from active list, calls CSwooshEmitter destructor
	}

	/* loop through swooshes */
	if (!SwooshMgr.SwooshActive.pHead)
		return;

	const Matrix4x4 &worldToScreen = g_cameraDirector.CurrentCamera().WorldToScreen();

	ts_Swoosh *pSwoosh = (ts_Swoosh *) SwooshMgr.SwooshActive.pHead;
	while (pSwoosh)
	{
		/* advance swoosh-- this does everything necessary to render swoosh */

		Swoosh__Advance(worldToScreen, pSwoosh, fFrameSeconds);

		/* to next swoosh */

		pSwoosh = pSwoosh->pNext;
	}
}

/* render all active swooshes */

void SwooshMgr__Render()
{
	/* quick exit */

	if (SwooshMgr.SwooshActive.pHead == NULL)
		return;

	/* may need to download particle texture set (swoosh data comes in with texture sets) */

	ParticleMgr__LockTextureSet(0);
#if defined(PS2) || defined(_XBOX)
	TextureMgr::GetBackBuffer()->Lock();
#else
	TextureMgr::GetWhiteTexture()->Lock();
#endif

	/* swooshes are pretty transparent */

	RenderState__Request(RS_ZWRITE, RS_FALSE);
	RenderState__Lock(RS_ZWRITE);

	// Culling is fubar'd (probably because of the -1 scale in the viewport), so I have to put this here.

	RenderState__Request(RS_CULL, RS_CULL_NONE );
	RenderState__Lock(RS_CULL);

	/* loop through swooshes */

	ts_Swoosh *pNextSwoosh = (ts_Swoosh *) SwooshMgr.SwooshActive.pHead;
	while (pNextSwoosh)
	{
		/* pre-load next so i can use continue */

		ts_Swoosh *pSwoosh = pNextSwoosh;
		pNextSwoosh = pSwoosh->pNext;

		/* let strip buffer render itself-- only if active and visible */

		if ((pSwoosh->u16Flags & (SWOOSH_ACTIVE | SWOOSH_VISIBLE)) != (SWOOSH_ACTIVE | SWOOSH_VISIBLE))
			continue;

		/* bail if outside constant alpha is at 0 */

		if ((pSwoosh->pfConstantAlpha) && (*pSwoosh->pfConstantAlpha == 0.0f))
			continue;

		/* in view? */

		if (Frustum__IntersectsWorldBox(SceneMgr::GetCurrentSceneView()->GetViewFrustum(), &pSwoosh->WorldBoundingBox) == false)
			continue;

		/* render */

		pSwoosh->pStripBuffer->Render(g4Identity, &g4Identity, 0);	// we know that we have one texture layer, so texture matrices array can be a single matrix
	}

	/* unlock */

#if defined(PS2) || defined(_XBOX)
	TextureMgr::GetBackBuffer()->Unlock();
#else
	TextureMgr::GetWhiteTexture()->Unlock();
#endif
	ParticleMgr__UnLockTextureSet(0);

	RenderState__Unlock(RS_CULL);
	RenderState__Unlock(RS_ZWRITE);
}

/* for pigpen, return memory address of requested swoosh source record. this is so i can update swoosh definition
in real time from pigpen. swoosh data is include with particle .dbl. so ask particle manager */

void *SwooshMgr__GetSwooshSource(int nDataIndex, int nEntryIndex)
{
	return(ParticleMgr__GetSwooshSource(nDataIndex, nEntryIndex));
}
void *SwooshMgr__GetSwooshEmitterSource(int nDataIndex, int nEntryIndex)
{
	return(ParticleMgr__GetSwooshEmitterSource(nDataIndex, nEntryIndex));
}

/************************** swoosh emitters ****************************/

/* constructor */

CSwooshEmitter::CSwooshEmitter()
{
	d_pSwooshHandle = NULL;
	d_pvEndpoint0 = NULL;
	d_pvEndpoint1 = NULL;

	/* cylinder emitter */

	d_pfSwooshHeight = NULL;
	d_pfSwooshAngle = NULL;
	d_pfSwooshRadius = NULL;
	d_pfSwooshYVel = NULL;
	d_pfSwooshXDelta = NULL;
	d_pfSwooshYDelta = NULL;
	d_pfSwooshLifetime = NULL;
}

/* destructor */

CSwooshEmitter::~CSwooshEmitter()
{
	if (d_pSwooshHandle)
	{
		for (int i = 0; i < d_nSwooshCount; i++)
			Swoosh__Kill(d_pSwooshHandle[i]);
		memFree(d_pSwooshHandle);
		d_pSwooshHandle = NULL;
	}
	if (d_pvEndpoint0)
	{
		memFree(d_pvEndpoint0);
		d_pvEndpoint0 = NULL;
	}
	if (d_pvEndpoint1)
	{
		memFree(d_pvEndpoint1);
		d_pvEndpoint1 = NULL;
	}
	if (d_pfSwooshHeight)
	{
		memFree(d_pfSwooshHeight);
		d_pfSwooshHeight = NULL;
	}
	if (d_pfSwooshAngle)
	{
		memFree(d_pfSwooshAngle);
		d_pfSwooshAngle = NULL;
	}
	if (d_pfSwooshRadius)
	{
		memFree(d_pfSwooshRadius);
		d_pfSwooshRadius = NULL;
	}
	if (d_pfSwooshYVel)
	{
		memFree(d_pfSwooshYVel);
		d_pfSwooshYVel = NULL;
	}
	if (d_pfSwooshXDelta)
	{
		memFree(d_pfSwooshXDelta);
		d_pfSwooshXDelta = NULL;
	}
	if (d_pfSwooshYDelta)
	{
		memFree(d_pfSwooshYDelta);
		d_pfSwooshYDelta = NULL;
	}
	if (d_pfSwooshLifetime)
	{
		memFree(d_pfSwooshLifetime);
		d_pfSwooshLifetime = NULL;
	}
}

/* intialize a new swoosh emitter-- set common data shared among different emitter types */

void CSwooshEmitter::Initialize(
		ts_SwooshEmitterSource *pSource,
		int nDataIndex,		// which of possibly-multiply-loaded particle .dbl's this one comes from
		Vector3 *pvPos,		// pointer to world position of emitter. if not stationary, must be persistent
		bool bStationary)		// if true, emitter does not move and we copy input position
{
	/* record data index */

	d_nDataIndex = nDataIndex;

	/* record position */

	if (bStationary)
	{
		d_pvPos = &d_vPos;
		d_vPos = *pvPos;
	}
	else
		d_pvPos = pvPos;

	/* lifetime, LOD */

	d_fLifetime = SwooshMgr__fRandomValue(pSource->fLifetime, pSource->u8LifetimeRandomPct);
	d_fLODSquared = pSource->fLOD * pSource->fLOD;

	/* flags */

	d_u32Flags = pSource->u16Flags & SWOOSH_EMITTER_SOURCE_TYPE_MASK;	// type
	d_u32Flags |= SWOOSH_EMITTER_ACTIVE | SWOOSH_EMITTER_ALL_SWOOSHES_INACTIVE;

	/* switch on type for final initialization */

	switch(d_u32Flags & SWOOSH_EMITTER_SOURCE_TYPE_MASK)
	{
	case SWOOSH_EMITTER_SOURCE_CYLINDER:
		InitializeCylinder(pSource);
		break;
	case SWOOSH_EMITTER_SOURCE_PREVIEW:
		InitializePreview(pSource);
		break;
	}
}

/* create a cylinder-type swoosh emitter */

void CSwooshEmitter::InitializeCylinder(ts_SwooshEmitterSource *pSource)
{
	/* record other data */

	d_fRadius = pSource->fCylinderRadius;
	d_nRadiusRandomPct = pSource->u8CylinderRadiusRandomPct;
	d_fHeight = pSource->fCylinderHeight;
	d_fSwooshYVel = pSource->fCylinderSwooshYVel;
	d_nSwooshYVelRandomPct = pSource->u8CylinderSwooshYVelRandomPct;
	d_fSwooshRotVel = pSource->fCylinderSwooshRotVel;
	d_fSwooshWidth = pSource->fCylinderSwooshWidth;
	d_nSwooshWidthRandomPct = pSource->u8CylinderSwooshWidthRandomPct;
	d_fSwooshLife = pSource->fCylinderSwooshLife;
	d_nSwooshLifeRandomPct = pSource->u8CylinderSwooshLifeRandomPct;
	d_nPointsOnCircle = pSource->u32CylinderPointsOnCircle;
	d_fDeltaTime = pSource->fCylinderDeltaTime;
	d_fDeltaAngle = pSource->fCylinderDeltaAngle;

	/* some checks */

	ASSERT(d_nPointsOnCircle != 0);

	/* we need access to the data for the swooshes that this emitter emits */

	ts_SwooshSource *pSwooshSource = (ts_SwooshSource *) ParticleMgr__GetSwooshSource(d_nDataIndex, pSource->u16SwooshID);
	float fPointLifetime = CONVERT_U8_TO_FLOAT(pSwooshSource->u8PointLifetime, MAX_S_LIFETIME);

	/* how many swooshes do we need? we can calculate how long it will take a swoosh to travel to the top of the */
	/* cylinder and then fade out */

	float fTimeToTop = (d_fHeight / d_fSwooshYVel) + fPointLifetime;

	/* now how many times will we launch a new set of swooshes during this time? */

	d_nSwooshCount = d_nPointsOnCircle;	// start with initial set
	while (fTimeToTop > 0.0f)
	{
		d_nSwooshCount += d_nPointsOnCircle;	// launch another set before original set dies
		fTimeToTop -= d_fDeltaTime;
	}
	CreateSwooshes(pSource);

	/* set up for swoosh creation/tracking */

	d_pfSwooshHeight = (float *) memAlloc(d_nSwooshCount * sizeof(float));
	ASSERT(d_pfSwooshHeight);	// shouldn't run out of memory!
	d_pfSwooshAngle = (float *) memAlloc(d_nSwooshCount * sizeof(float));
	ASSERT(d_pfSwooshAngle);	// shouldn't run out of memory!
	d_pfSwooshRadius = (float *) memAlloc(d_nSwooshCount * sizeof(float));
	ASSERT(d_pfSwooshRadius);	// shouldn't run out of memory!
	d_pfSwooshYVel = (float *) memAlloc(d_nSwooshCount * sizeof(float));
	ASSERT(d_pfSwooshYVel);	// shouldn't run out of memory!
	d_pfSwooshXDelta = (float *) memAlloc(d_nSwooshCount * sizeof(float));
	ASSERT(d_pfSwooshXDelta);	// shouldn't run out of memory!
	d_pfSwooshYDelta = (float *) memAlloc(d_nSwooshCount * sizeof(float));
	ASSERT(d_pfSwooshYDelta);	// shouldn't run out of memory!
	d_pfSwooshLifetime = (float *) memAlloc(d_nSwooshCount * sizeof(float));
	ASSERT(d_pfSwooshLifetime);	// shouldn't run out of memory!

	/* emitter final set up */

	d_u32Flags &= ~SWOOSH_EMITTER_TYPE_MASK;
	d_u32Flags |= SWOOSH_EMITTER_CYLINDER;
	d_fTimer = 0.0f;
	d_nSwooshIndex = 0;
	d_fAngle = 0.0f;
}

/* create a preview emitter-- for pigpen use to preview swooshes directly without a defined emitter-- this emitter
just creates 3 swooshes that travel in circles at right angles to each other */

void CSwooshEmitter::InitializePreview(ts_SwooshEmitterSource *pSource)
{
	/* this emitter creates 3 swooshes */

	d_nSwooshCount = 3;
	CreateSwooshes(pSource);

	/* emitter final set up */

	d_u32Flags &= ~SWOOSH_EMITTER_TYPE_MASK;
	d_u32Flags |= SWOOSH_EMITTER_PREVIEW;
	d_fTimer = 0.0f;
}

/* create a set of swooshes for this emitter. swooshes take a lot of handles (they are vertex/strip buffers), and each
emitter may require lots of them, so we have a system in place where the user can limit the number of emitters of a
particular type that actually create the swooshes. since all emitters of a particular type create the same group
of swooshes (same count, same source), one group of swooshes can be passed from emitter to emitter as needed. this
is typically based on whether the emitter needs is active or not-- if an active emitter finds that it doesn't have
any swooshes, it looks through the list of active swooshes looking for an inactive emitter of the same type. if found,
it takes the swoosh group */

void CSwooshEmitter::CreateSwooshes(ts_SwooshEmitterSource *pSource)
{
	/* reached max allowed? */

	int i;
	for (i = 0; i < MAX_SHARED_SWOOSH_GROUP_EMITTERS; i++)
	{
		if (pSource == SwooshMgr.d_pSharedSwooshGroupEmitterSource[i])
			break;
	}
	if (i == MAX_SHARED_SWOOSH_GROUP_EMITTERS)
		d_nSharedSwooshGroupID = -1;	// not found in list, not restricted
	else
	{
		/* this emitter is in restricted list. have we reached limit? */

		d_nSharedSwooshGroupID = i;	// this lets emitter look for other emitters of its own kind
		if (SwooshMgr.d_nSharedSwooshGroups[i] >= SwooshMgr.d_nMaxSharedSwooshGroups[i])
		{
			/* this emitter is not allowed to create a swoosh group-- it will have to poach somebody else's */

			ASSERT(d_pSwooshHandle == NULL);
			ASSERT(d_pvEndpoint0 == NULL);
			ASSERT(d_pvEndpoint1 == NULL);
			return;
		}

		/* will create a new swoosh group-- this counts toward reaching limit */

		SwooshMgr.d_nSharedSwooshGroups[i]++;
	}

	/* malloc a list */

	d_pSwooshHandle = (t_Handle *) memAlloc(d_nSwooshCount * sizeof(t_Handle));
	ASSERT(d_pSwooshHandle);	// shouldn't run out of memory!
	d_pvEndpoint0 = (Vector3Packed *) memAlloc(d_nSwooshCount * sizeof(Vector3Packed));
	ASSERT(d_pvEndpoint0);	// shouldn't run out of memory!
	d_pvEndpoint1 = (Vector3Packed *) memAlloc(d_nSwooshCount * sizeof(Vector3Packed));
	ASSERT(d_pvEndpoint1);	// shouldn't run out of memory!

	/* initialize swooshes */

	for (i = 0; i < d_nSwooshCount; i++)
	{
		d_pSwooshHandle[i] = SwooshMgr__CreateSwooshFromRecord(
										d_nDataIndex,
										pSource->u16SwooshID,
										&d_pvEndpoint0[i],
										&d_pvEndpoint1[i],
										DEFAULT_POINT_COUNT_MULTIPLIER);
		ASSERT(d_pSwooshHandle[i] != INVALID_HANDLE);	// if this hits, we need more swooshes
	}
}

/* advance a swoosh emitter. return false if it dies */

bool CSwooshEmitter::Advance(const float fFrameSeconds)
{
	/* active? */

	if ((d_u32Flags & SWOOSH_EMITTER_ACTIVE) == 0)
		return(true);

	/* common stuff */

	if (d_fLifetime > 0.0f)
	{
		d_fLifetime -= fFrameSeconds;
		if (d_fLifetime < 0.0f)
			return(false);
	}

	/* check LOD */

	float fBestDistanceSquared = MAXFLOAT;
	CScene *pScene = SceneMgr::GetFirstScene();
	while (pScene)
	{
		/* vCameraInWorldPos is a frame old, but should be ok for our purposes */

		float fDistanceSquared = (pScene->d_cameraWorldPos - *d_pvPos).LengthSquared();
		if (fDistanceSquared < fBestDistanceSquared)
			fBestDistanceSquared = fDistanceSquared;
		pScene = SceneMgr::GetNextScene(pScene);
	}
	if (fBestDistanceSquared > d_fLODSquared)
	{
		/* beyond lod range, turn off all swooshes and skip advancement */

		DeactivateAllSwooshes();
		return(true);
	}

	/* this emitter should advance. if it has no swoosh group, see if it can steal one from another emitter */
	/* of the same kind that is not advancing. the intent is that this code should always find a swoosh set */
	/* to steal-- the number of swoosh sets should be at least as great as the number of swoosh emitters that */
	/* are visible at once */

	if (d_pSwooshHandle == NULL)
	{
		/* look through emitters for one that is inactive or marked as all-swooshes-inactive */

		ASSERTS(d_nSharedSwooshGroupID != -1, "Swoosh emitter has no swoosh group, but not marked as restricted");
		CSwooshEmitter *pBestSwooshEmitter = NULL;
		CSwooshEmitter *pSwooshEmitter = SwooshMgr.d_EmitterList->ActiveHead();
		while (pSwooshEmitter)
		{
			/* if this emitter has a swoosh group and is inactive or marked as all-swooshes-inactive, consider */
			/* stealing swoosh group. ok to check itself since it doesn't have a swoosh group and will fail test */

			if ((pSwooshEmitter->d_nSharedSwooshGroupID == d_nSharedSwooshGroupID) && (pSwooshEmitter->d_pSwooshHandle))
			{
				/* swoosh of same type that has a swoosh group-- can we steal it? */

				if ((pSwooshEmitter->d_u32Flags & SWOOSH_EMITTER_ACTIVE) == 0)
				{
					pBestSwooshEmitter = pSwooshEmitter;
					break;
				}
				if (pSwooshEmitter->d_u32Flags & SWOOSH_EMITTER_ALL_SWOOSHES_INACTIVE)
				{
					if (pBestSwooshEmitter == NULL)
						pBestSwooshEmitter = pSwooshEmitter;	// no best candidate yet, take this one
					else if (pSwooshEmitter->d_nDeactivatedID < pBestSwooshEmitter->d_nDeactivatedID)
						pBestSwooshEmitter = pSwooshEmitter;	// this emitter was deactivated longer ago, it's a better candidate
				}
			}

			/* to next emitter */

			pSwooshEmitter = pSwooshEmitter->next;
		}

		/* did we find one? if not, no swoosh group to steal-- don't advance */

		if (pBestSwooshEmitter == NULL)
		{
			ASSERTS(false, "Couldn't find a swoosh group to steal-- increase limit");
			return(true);
		}

		/* steal swoosh group */

		d_pSwooshHandle = pBestSwooshEmitter->d_pSwooshHandle;
		d_pvEndpoint0 = pBestSwooshEmitter->d_pvEndpoint0;
		d_pvEndpoint1 = pBestSwooshEmitter->d_pvEndpoint1;
		pBestSwooshEmitter->d_pSwooshHandle = NULL;
		pBestSwooshEmitter->d_pvEndpoint0 = NULL;
		pBestSwooshEmitter->d_pvEndpoint1 = NULL;
	}

	/* we assume that if a swoosh emitter is advancing, it has at least one active swoosh. no real penalty here, */
	/* even if this bit is set incorrectly, the only thing that will happen is that DeactivateAllSwooshes will run */
	/* once */

	d_u32Flags &= ~SWOOSH_EMITTER_ALL_SWOOSHES_INACTIVE;

	/* advance correct type of emitter */

	switch(d_u32Flags & SWOOSH_EMITTER_TYPE_MASK)
	{
	case SWOOSH_EMITTER_CYLINDER:
		return(AdvanceCylinder(fFrameSeconds));
	case SWOOSH_EMITTER_PREVIEW:
		return(AdvancePreview(fFrameSeconds));
	default:
		ASSERT(false);	// unhandled case
		return(true);
	}
}

/* advance a cylinder swoosh emitter */

bool CSwooshEmitter::AdvanceCylinder(const float fFrameSeconds)
{
	/* time to start out new swoosh(es)? */

	d_fTimer -= fFrameSeconds;
	if (d_fTimer <= 0.0f)
	{
		/* start them up */

		float fAngle = d_fAngle;
		float fAngleDelta = Math::TwoPi / (float) d_nPointsOnCircle;
		for (int i = 0; i < d_nPointsOnCircle; i++, fAngle += fAngleDelta)
		{
			/* get swoosh to start, deactivate, and activate again (clears history) */

			Swoosh__Deactivate(d_pSwooshHandle[d_nSwooshIndex], true);
			Swoosh__Activate(d_pSwooshHandle[d_nSwooshIndex]);

			/* start it at bottom of cylinder, unless this is a sparse cylinder */

			if (d_fSwooshLife == 0.0f)
			{
				d_pfSwooshHeight[d_nSwooshIndex] = 0.0f;
				d_pfSwooshAngle[d_nSwooshIndex] = fAngle;
				d_pfSwooshRadius[d_nSwooshIndex] = SwooshMgr__fRandomValue(d_fRadius, d_nRadiusRandomPct);
				d_pfSwooshYVel[d_nSwooshIndex] = SwooshMgr__fRandomValue(d_fSwooshYVel, d_nSwooshYVelRandomPct);
				d_pfSwooshLifetime[d_nSwooshIndex] = 0.0f;
			}
			else
			{
				/* sparse cylinder-- swooshes start at random point along cylinder and last for a limited time */

				d_pfSwooshHeight[d_nSwooshIndex] = SwooshRandom.UpTo(d_fHeight);	// 0.0 to cylinder height
				d_pfSwooshAngle[d_nSwooshIndex] = SwooshRandom.UpTo(Math::TwoPi);	// 0.0 to 2PI
				d_pfSwooshRadius[d_nSwooshIndex] = SwooshMgr__fRandomValue(d_fRadius, d_nRadiusRandomPct);
				d_pfSwooshYVel[d_nSwooshIndex] = SwooshMgr__fRandomValue(d_fSwooshYVel, d_nSwooshYVelRandomPct);
				d_pfSwooshLifetime[d_nSwooshIndex] = SwooshMgr__fRandomValue(d_fSwooshLife, d_nSwooshLifeRandomPct);
			}

			/* must have some upward velocity! */

			if (d_pfSwooshYVel[d_nSwooshIndex] <= 0.0f)
				d_pfSwooshYVel[d_nSwooshIndex] = 0.1f;

			/* width of swoosh can be randomized. we want leading edge of swoosh to be perpendicular to its direction */
			/* of travel. calculate her at angle == 0, transform according to angle at advance */

			float fSwooshHalfWidth = 0.5f * SwooshMgr__fRandomValue(d_fSwooshWidth, d_nSwooshWidthRandomPct);

			/* calculate horizontal velocity around edge of cylinder in meters/second. we know that circumference */
			/* is 2 * pi * r. every second we wil travel (circumference * (radians/second) / (2 * pi)), which */
			/* reduces down to radius * radians/second */

			float fOrbitVelocity = d_pfSwooshRadius[d_nSwooshIndex] * d_fSwooshRotVel;

			/* now we can get an angle of travel as though the cylinder were unrolled into a flat sheet. since we */
			/* always have some up velocity, value should be between -90 and 90 degrees (but in radians, of course) */
			/* where 0 degrees is straight up */

			float fTravelAngle = Math::ArcTan2(-fOrbitVelocity, d_pfSwooshYVel[d_nSwooshIndex]);
			float fCos;
			float fSin;
			Math::SinCos(fTravelAngle, fSin, fCos);
			d_pfSwooshXDelta[d_nSwooshIndex] = fSwooshHalfWidth * fCos;
			d_pfSwooshYDelta[d_nSwooshIndex] = fSwooshHalfWidth * fSin;

			/* to next swoosh in list */

			d_nSwooshIndex++;
			if (d_nSwooshIndex == d_nSwooshCount)
				d_nSwooshIndex = 0;
		}

		/* update for next time */

		d_fTimer += d_fDeltaTime;
		d_fAngle = Math::NormalAngle(d_fAngle + d_fDeltaAngle);
	}

	/* update all active swooshes */

	for (int i = 0; i < d_nSwooshCount; i++)
	{
		/* point to swoosh, skip if not active */

		ts_Swoosh *pSwoosh = SWOOSH_GET_RECORD(d_pSwooshHandle[i]);
		if ((pSwoosh->u16Flags & SWOOSH_ACTIVE) == 0)
			continue;

		/* move it and calculate center point */

		d_pfSwooshHeight[i] += d_pfSwooshYVel[i] * fFrameSeconds;
		d_pfSwooshAngle[i] += d_fSwooshRotVel * fFrameSeconds;
		float fCos;
		float fSin;
		Math::SinCos(d_pfSwooshAngle[i], fSin, fCos);
		Vector3 vCenter = *d_pvPos;
		vCenter.Set(vCenter.x() + d_pfSwooshRadius[i] * fSin,
						vCenter.y() + d_pfSwooshHeight[i],
						vCenter.z() + d_pfSwooshRadius[i] * fCos);

		/* calculate new endpoints. rotate delta from center point in x/y plane so that it is tangent to wall of cyclinder */

		Vector3 vDelta(d_pfSwooshXDelta[i] * fCos, d_pfSwooshYDelta[i], d_pfSwooshXDelta[i] * -fSin);
		d_pvEndpoint0[i] = vCenter + vDelta;
		d_pvEndpoint1[i] = vCenter - vDelta;

		/* reached top of cylinder? */

		if (d_pfSwooshHeight[i] >= d_fHeight)
			Swoosh__Deactivate(d_pSwooshHandle[i], false);	// deactivate, but allow it to fade out

		/* maybe time to die? */

		if (d_pfSwooshLifetime[i] != 0.0f)
		{
			d_pfSwooshLifetime[i] -= fFrameSeconds;
			if (d_pfSwooshLifetime[i] <= 0.0f)
				Swoosh__Deactivate(d_pSwooshHandle[i], false);	// deactivate, but allow it to fade out
		}
	}

	/* still alive */

	return(true);
}

/* advance a preview swoosh emitter-- for pigpen, to preview swooshes directly without defining an emitter. three
swooshes rotating around a center point at right angles to each other */

bool CSwooshEmitter::AdvancePreview(const float fFrameSeconds)
{
	/* just keep them turning */

#define PREVIEW_SWOOSH_EMITTER_ROTATION_VEL			3.0f
#define PREVIEW_SWOOSH_EMITTER_ENDPOINT0_DISTANCE	3.0f
#define PREVIEW_SWOOSH_EMITTER_ENDPOINT1_DISTANCE	2.0f

	d_fTimer += fFrameSeconds;
	float fAngle = Math::NormalAngle(d_fTimer * PREVIEW_SWOOSH_EMITTER_ROTATION_VEL);
	float fCos = Math::Cos(fAngle);
	float fSin = Math::Sin(fAngle);
	d_pvEndpoint0[0].Set(
		d_pvPos->x() + (fSin * PREVIEW_SWOOSH_EMITTER_ENDPOINT0_DISTANCE),
		d_pvPos->y(),
		d_pvPos->z() + (fCos * PREVIEW_SWOOSH_EMITTER_ENDPOINT0_DISTANCE));
	d_pvEndpoint1[0].Set(
		d_pvPos->x() + (fSin * PREVIEW_SWOOSH_EMITTER_ENDPOINT1_DISTANCE),
		d_pvPos->y(),
		d_pvPos->z() + (fCos * PREVIEW_SWOOSH_EMITTER_ENDPOINT1_DISTANCE));
	d_pvEndpoint0[1].Set(
		d_pvPos->x(),
		d_pvPos->y() + (fSin * PREVIEW_SWOOSH_EMITTER_ENDPOINT0_DISTANCE),
		d_pvPos->z() + (fCos * PREVIEW_SWOOSH_EMITTER_ENDPOINT0_DISTANCE));
	d_pvEndpoint1[1].Set(
		d_pvPos->x(),
		d_pvPos->y() + (fSin * PREVIEW_SWOOSH_EMITTER_ENDPOINT1_DISTANCE),
		d_pvPos->z() + (fCos * PREVIEW_SWOOSH_EMITTER_ENDPOINT1_DISTANCE));
	d_pvEndpoint0[2].Set(
		d_pvPos->x() + (fSin * PREVIEW_SWOOSH_EMITTER_ENDPOINT0_DISTANCE),
		d_pvPos->y() + (fCos * PREVIEW_SWOOSH_EMITTER_ENDPOINT0_DISTANCE),
		d_pvPos->z());
	d_pvEndpoint1[2].Set(
		d_pvPos->x() + (fSin * PREVIEW_SWOOSH_EMITTER_ENDPOINT1_DISTANCE),
		d_pvPos->y() + (fCos * PREVIEW_SWOOSH_EMITTER_ENDPOINT1_DISTANCE),
		d_pvPos->z());

	/* activate swooshes if inactive-- may be coming into lod range */

	for (int i = 0; i < d_nSwooshCount; i++)
		Swoosh__Activate(d_pSwooshHandle[i]);

	/* still alive */

	return(true);
}

/* activate/deactivate */

void CSwooshEmitter::Activate()
{
	d_u32Flags |= SWOOSH_EMITTER_ACTIVE;
}
void CSwooshEmitter::Deactivate()
{
	d_u32Flags &= ~SWOOSH_EMITTER_ACTIVE;
	DeactivateAllSwooshes();
}

/* turn off all emitter's swooshes */

void CSwooshEmitter::DeactivateAllSwooshes()
{
	/* maybe already there */

	if (d_u32Flags & SWOOSH_EMITTER_ALL_SWOOSHES_INACTIVE)
		return;
	d_u32Flags |= SWOOSH_EMITTER_ALL_SWOOSHES_INACTIVE;
	d_nDeactivatedID = SwooshMgr.d_nDeactivatedID++;

	/* loop through swooshes */

	for (int i = 0; i < d_nSwooshCount; i++)
	{
		ts_Swoosh *pSwoosh = SWOOSH_GET_RECORD(d_pSwooshHandle[i]);
		if (pSwoosh->u16Flags & SWOOSH_ACTIVE)
			Swoosh__Deactivate(pSwoosh->Handle, false);	// let the swoosh fade out
	}
}

/************************** deal with individual swooshes-- internal to manager ****************************/

/* create a swoosh. this routine does most of the work, but leaves undone the method of following swoosh's movement */

ts_Swoosh *Swoosh__Create(
							ts_SwooshSource *pSwooshSource,
							int nDataIndex,
							float fPointCountMultiplier,
							int nForcePointCount,			// for ballistic swoosh support, 0 means ignore
							/* the following arguments are for swooshes without source data-- created directly */
							ts_bRGBA *pColor /*= NULL*/,
							float fMaxAlpha /*= 0.0f*/,
							float fPointLifetime /*= 0.0f*/,
							bool bAdditiveRender /*= false*/,
							float fLODDistance /*= 0.0f*/)
{
	int i;

	/* alloc a swoosh-- should always succeed-- steals oldest if none available */

	ts_Swoosh *pSwoosh = SwooshMgr__AllocSwoosh();
	if (pSwoosh == NULL)
		return(NULL);

	/* initial record state */

	pSwoosh->u16Flags = 0;
	pSwoosh->nDataIndex = nDataIndex;
	pSwoosh->pSwooshSource = pSwooshSource;
	if (pColor)
		pSwoosh->Color = *pColor;	// for very simple swooshes, only used if pSwooshSource is NULL
	pSwoosh->fMaxAlpha = 255.0f * fMaxAlpha;	// 0..255 range, but as float. for very simple swooshes, only used if pSwooshSource is NULL
	pSwoosh->pfColorMult = NULL;
	pSwoosh->fColorMultFloor = 0.0f;
	pSwoosh->pfConstantAlpha = NULL;

	/* number of data points for this swoosh depends on lifetime-- lifetime is the length of time it takes for a new */
	/* point to fade out to nothing. the point count multiplier can leave room for intermediate points that may be */
	/* needed for fast-moving swoosh sources */

	if (pSwooshSource)
		pSwoosh->fPointLifetime = CONVERT_U8_TO_FLOAT(pSwooshSource->u8PointLifetime, MAX_S_LIFETIME);
	else
		pSwoosh->fPointLifetime = fPointLifetime;
	int nPoints = (int) (fPointCountMultiplier * 30.0f * pSwoosh->fPointLifetime);	// assume 30 hz, leave some room for intermediate points
	if (nForcePointCount != 0)
		nPoints = nForcePointCount;	// override with forced value
	if (fPointCountMultiplier == 1.0f)
		pSwoosh->u16Flags |= SWOOSH_NO_INTERMEDIATE_POINTS;

	/* fLODDistance is distance, in meters, at which swoosh disappears */

	if (pSwooshSource)
		pSwoosh->fLODDistance = CONVERT_U8_TO_FLOAT(pSwooshSource->u8LOD, MAX_S_LOD);
	else
		pSwoosh->fLODDistance = fLODDistance;

	/* source data, if any, overrides passed bAdditiveRender */

	if (pSwooshSource)
		bAdditiveRender = (pSwooshSource->u32Flags & SWOOSH_ADDITIVE_RENDER) != 0;

	/* we need a history buffer for points */

	MEM_SET_ALLOC_NAME("Swoosh history");
	pSwoosh->pSwooshPoints = (ts_SwooshPoint *) memAlloc(nPoints * sizeof(ts_SwooshPoint));
	pSwoosh->nSwooshPointCount = nPoints;
	pSwoosh->nSwooshPointIndex = 0;	// buffer is circular

	/* create a vertex and strip buffer to show generated geometry. each new point contributes 2 vertices */

	int nVertices = nPoints * 2;
	MEM_SET_ONE_SHOT_NEW_NAME("Swoosh vtx buffer");
	pSwoosh->pVtxBuffer = new VtxBuffer;
	MEM_SET_ONE_SHOT_NEW_NAME("Swoosh strip buffer");
	pSwoosh->pStripBuffer = new StripBuffer;

	/* set up the vertex buffer. geometry using the VtxBuffer class must be textured. swooshes are currently */
	/* single-textured */

	u32 u32TextureFlags[VTX_BUFFER_TEXTURE_LAYERS] = {0};
	u32TextureFlags[0] = VtxBuffer::VERT_TEX_DIM2;
	pSwoosh->pVtxBuffer->Init(nVertices, VtxBuffer::VERT_HAS_DIFFUSE, u32TextureFlags);

	/* set the uv coordinates. for now we just use the top row of pixels in the texture */

	pSwoosh->pVtxBuffer->Lock();
	for (i = 0; i < nVertices; i += 2)
	{
		static Vector2 vEdge1TexCoord(0.0f, 0.0f);	// left edge of texture
		static Vector2 vEdge2TexCoord(1.0f, 0.0f);	// right edge of texture
		pSwoosh->pVtxBuffer->TexCoordv2(0, i + 0) = vEdge1TexCoord;
		pSwoosh->pVtxBuffer->TexCoordv2(0, i + 1) = vEdge2TexCoord;
	}
	pSwoosh->pVtxBuffer->UnLock();

	/* get bind-- no support currently for animating textures */

	Texture* pTexture;
	if (pSwooshSource)
		pTexture = pSwooshSource->pTexture;
	else
	{
#if defined(PS2) || defined(_XBOX)
		pTexture = TextureMgr::GetBackBuffer();
		pSwoosh->u16Flags |= SWOOSH_IS_BLUR;
#else
		pTexture = TextureMgr::GetWhiteTexture();
#endif
	}

	/* edge 2 multpliers */

	pSwoosh->fEdge2ColorMult = 1.0f;
	pSwoosh->fEdge2AlphaMult = 1.0f;
	if (pSwooshSource)
	{
		pSwoosh->fEdge2ColorMult = CONVERT_U8_TO_FLOAT(pSwooshSource->u8Edge2ColorMult, 1.0f);
		pSwoosh->fEdge2AlphaMult = CONVERT_U8_TO_FLOAT(pSwooshSource->u8Edge2AlphaMult, 1.0f);
	}

	/* initialize the strip buffer */
	u8 method;
	u8 wrap;
	u8 blend;
	if ((pSwoosh->u16Flags & SWOOSH_IS_BLUR) == 0)
	{
		method = TS_METHOD_MODULATE;
		wrap = TS_WRAP_REPEAT;
		blend = bAdditiveRender ? RS_BLENDFUNC_ADDITIVE : RS_BLENDFUNC_DEFAULT;
	}
	else
	{
		method = TS_METHOD_REPLACE;
		wrap = TS_WRAP_CLAMP;
		blend = RS_BLENDFUNC_DEFAULT;
	}

	pSwoosh->pStripBuffer->Init();

	pSwoosh->pStripBuffer->SetPassTexture(
		0,																						// u32 passNumber
		pTexture,																			// s16 bind
		method,																				// u8 method
		wrap,																					// u8 wrapU
		wrap,																					// u8 wrapV
		TS_FILTER_BILINEAR,																// u8 filter
		blend);																				// u8 blend
	pSwoosh->pStripBuffer->SetVtxBuffer(pSwoosh->pVtxBuffer);

#ifdef PS2
	if ((pSwoosh->u16Flags & SWOOSH_IS_BLUR) == 0)
		pSwoosh->pStripBuffer->Overbrighting(bAdditiveRender);
#endif

	/* add a single strip to represent the swoosh. the strip is constant length, and always uses the vertices from the */
	/* vertex buffer in the same order-- if we only need a portion of it, we will set up the vertices to build degenerate */
	/* triangles. the vertex buffer needs to be completely re-built each frame anyway, since the vertices of the swoosh are */
	/* always changing */

	MEM_SET_ALLOC_NAME("Swoosh strip indices");
	u16 *pu16StripIndicies = (u16 *) memAlloc(nVertices * sizeof(u16));
	for (i = 0; i < nVertices; i++)
		pu16StripIndicies[i] = i;
	pSwoosh->pStripBuffer->AddStrip(pu16StripIndicies, nVertices);
	memFree(pu16StripIndicies);

	/* default inactive */

	Swoosh__Deactivate(pSwoosh->Handle);

	/* done */

	return(pSwoosh);
}

/* advance a swoosh. this gets it all set to be rendered */

void Swoosh__Advance(const Matrix4x4 &worldToScreen, ts_Swoosh *pSwoosh, float fFrameSeconds)
{
	/* only if active */

	if ((pSwoosh->u16Flags & SWOOSH_ACTIVE) == 0)
		return;

	/* where is nearest camera (for LOD control)? we use bounding box for this test because it contains older points, */
	/* not just the last point. when a swoosh covers a lot of distance, we want to consider the nearest or all the */
	/* points for lod control (including overall alpha, otherwise the swoosh alphas out too fast as the current point */
	/* moves away from the camera but the old points are still near) */

	if (pSwoosh->u16Flags & SWOOSH_BOUNDING_BOX_VALID)
	{
		Vector3 vCenter;
		vCenter = BoundingBox__GetCenter(&pSwoosh->WorldBoundingBox);
		float fBestDistanceSquared = MAXFLOAT;
		CScene *pScene = SceneMgr::GetFirstScene();
		while (pScene)
		{
			/* vCameraInWorldPos is a frame old, but should be ok for our purposes */

			float fDistanceSquared = (pScene->d_cameraWorldPos - vCenter).LengthSquared();
			if (fDistanceSquared < fBestDistanceSquared)
				fBestDistanceSquared = fDistanceSquared;
			pScene = SceneMgr::GetNextScene(pScene);
		}

		/* adjust for radius of bounding box */

		pSwoosh->fDistanceToNearestCamera = Math::Sqrt(fBestDistanceSquared);


#if defined(GCN)
		// To avoid Strict FPU Check for GCN -BHY
		pSwoosh->WorldBoundingBox.Max.x( Math::Clamp(pSwoosh->WorldBoundingBox.Max.x(), -1.0e5f, 1.0e5f));
		pSwoosh->WorldBoundingBox.Max.y( Math::Clamp(pSwoosh->WorldBoundingBox.Max.y(), -1.0e5f, 1.0e5f));
		pSwoosh->WorldBoundingBox.Max.z( Math::Clamp(pSwoosh->WorldBoundingBox.Max.z(), -1.0e5f, 1.0e5f));

		pSwoosh->WorldBoundingBox.Min.x( Math::Clamp(pSwoosh->WorldBoundingBox.Min.x(), -1.0e5f, 1.0e5f));
		pSwoosh->WorldBoundingBox.Min.y( Math::Clamp(pSwoosh->WorldBoundingBox.Min.y(), -1.0e5f, 1.0e5f));
		pSwoosh->WorldBoundingBox.Min.z( Math::Clamp(pSwoosh->WorldBoundingBox.Min.z(), -1.0e5f, 1.0e5f));

		if( (Math::Zero(pSwoosh->WorldBoundingBox.Max.x())) ||
			(Math::Zero(pSwoosh->WorldBoundingBox.Max.y())) ||
			(Math::Zero(pSwoosh->WorldBoundingBox.Max.z())) )
		{
			pSwoosh->WorldBoundingBox.Max.Set( 0.1f, 0.1f, 0.1f);
		}
		if( (Math::Zero(pSwoosh->WorldBoundingBox.Min.x())) ||
			(Math::Zero(pSwoosh->WorldBoundingBox.Min.y())) ||
			(Math::Zero(pSwoosh->WorldBoundingBox.Min.z())) )
		{
			pSwoosh->WorldBoundingBox.Min.Set( -0.1f, -0.1f, -0.1f);
		}
#endif //GCN



		pSwoosh->fDistanceToNearestCamera -= 0.5f * (pSwoosh->WorldBoundingBox.Max - pSwoosh->WorldBoundingBox.Min).Length();
		if (pSwoosh->fDistanceToNearestCamera < 0.0f)
			pSwoosh->fDistanceToNearestCamera = 0.0f;	// distance shouldn't be negative

		/* bounding box is not valid until it gets rebuilt below */

		pSwoosh->u16Flags &= ~SWOOSH_BOUNDING_BOX_VALID;
	}
	else if (pSwoosh->u16Flags & SWOOSH_ADD_NEW_POINT)
	{
		/* we don't have a valid bounding box for swoosh, so just use current location-- this is an estimate */

		Vector3 pvPos;
		switch(pSwoosh->u16Flags & SWOOSH_TYPE_MASK)
		{
		case SWOOSH_TYPE_BONE:
			pvPos = pSwoosh->pCActor->GetBodyInWorld();
			break;
		case SWOOSH_TYPE_ENDPOINT:
			pvPos = *pSwoosh->pvEndpoints[0];
			break;
		case SWOOSH_TYPE_BALLISTIC:
			pvPos = pSwoosh->pBallisticPath->GetStartPos();
			break;
		default:
			ASSERT(false);	// unknown type
			break;
		}
		float fBestDistanceSquared = MAXFLOAT;
		CScene *pScene = SceneMgr::GetFirstScene();
		while (pScene)
		{
			/* vCameraInWorldPos is a frame old, but should be ok for our purposes */

			float fDistanceSquared = (pScene->d_cameraWorldPos - pvPos).LengthSquared();
			if (fDistanceSquared < fBestDistanceSquared)
				fBestDistanceSquared = fDistanceSquared;
			pScene = SceneMgr::GetNextScene(pScene);
		}
		pSwoosh->fDistanceToNearestCamera = Math::Sqrt(fBestDistanceSquared);
	}
	else
	{
		/* we don't have a valid bounding box, and swoosh isn't adding points-- don't worry about it */

		pSwoosh->fDistanceToNearestCamera = MAXFLOAT;
	}

	/* check lod distance */

	if (pSwoosh->fDistanceToNearestCamera >= pSwoosh->fLODDistance)
	{
		/* if we are still adding points, we'll just suspend the swoosh-- it might come back into lod range */

		if (pSwoosh->u16Flags & SWOOSH_ADD_NEW_POINT)
		{
			pSwoosh->u16Flags |= SWOOSH_LOD_SUSPENDED;
			pSwoosh->u16Flags &= ~SWOOSH_VISIBLE;
		}
		else
		{
			/* swoosh is just fading away (via call to Swoosh__Deactivate, with bHideImmediately = false), go ahead */
			/* and really deactivate it */

			Swoosh__Deactivate(pSwoosh->Handle, true);
		}
		return;	// no more processing
	}
	else
	{
		/* inside LOD distance. if it was suspended, and still adding points, re-activate-- but with a fresh history buffer-- */
		/* don't want gaps */

		if ((pSwoosh->u16Flags & (SWOOSH_LOD_SUSPENDED | SWOOSH_ADD_NEW_POINT)) == (SWOOSH_LOD_SUSPENDED | SWOOSH_ADD_NEW_POINT))
			Swoosh__Activate(pSwoosh->Handle);
	}

	/* maybe grab new history point */

	bool bAddNewPoint = (pSwoosh->u16Flags & SWOOSH_ADD_NEW_POINT) != 0;
	if (((pSwoosh->u16Flags & SWOOSH_TYPE_MASK) == SWOOSH_TYPE_BONE) && (pSwoosh->pCActor->GetBoneVisibility(pSwoosh->nBoneID) == false))
		bAddNewPoint = false;	// invisible bones don't have swooshes
	if (bAddNewPoint)
	{
		switch(pSwoosh->u16Flags & SWOOSH_TYPE_MASK)
		{
		case SWOOSH_TYPE_BONE:
		case SWOOSH_TYPE_ENDPOINT:
			Swoosh__AddPoint(pSwoosh, fFrameSeconds);
			break;
		case SWOOSH_TYPE_BALLISTIC:
			Swoosh__CalculateBallisticPath(pSwoosh, fFrameSeconds);
			break;
		default:
			ASSERT(false);	// unknown type
			break;
		}
	}

	/* update lifetimes for all points */

	for (int i = 0; i < pSwoosh->nSwooshPointCount; i++)
		pSwoosh->pSwooshPoints[i].fLifetime -= fFrameSeconds;

	/* color may be controlled by a multiplier, to tie swoosh color to collision color data */

	float fColorMult = 1.0f;
	if (pSwoosh->pfColorMult)
		fColorMult = Math::Max(*pSwoosh->pfColorMult, pSwoosh->fColorMultFloor);
	ts_bRGBA SwooshColor;
	if (pSwoosh->pSwooshSource == NULL)
	{
		/* simple swoosh (not pigpen-generated) uses the same color for the whole swoosh */

		SwooshColor = pSwoosh->Color;

		/* add in the color multiplier */

		SwooshColor.R = (u8) ((float) SwooshColor.R * fColorMult);
		SwooshColor.G = (u8) ((float) SwooshColor.G * fColorMult);
		SwooshColor.B = (u8) ((float) SwooshColor.B * fColorMult);
	}

	/* calculate an overall alpha value based on lod distance so that swoosh fades out as it gets farther away. there is */
	/* a potential problem with this for split screen-- since the swoosh has one vertex buffer, it will be rendered at */
	/* the same alpha value in both scenes, so it may be near in one scene and far away in another and render at the same */
	/* alpha level. oh well, at least it won't blink out, since as long as it is near one camera it won't be lod-suspended */

	float fAlphaMult = 1.0f;
	if (pSwoosh->fDistanceToNearestCamera > HIRES_LOD_RANGE)
	{
		/* fade from 1.0 at HIRES_LOD_RANGE to 0.0 at LOD distance */

		fAlphaMult = 1.0f - ((pSwoosh->fDistanceToNearestCamera - HIRES_LOD_RANGE) / (pSwoosh->fLODDistance - HIRES_LOD_RANGE));
		ASSERT(fAlphaMult > 0.0f);	// distance to camera should always be less than lod distance
	}

	/* add in constant alpha controlled by some outside source-- typically the actor that owns the swoosh. */
	/* this supports having the swoosh fade out along with the parent source */

	if (pSwoosh->pfConstantAlpha)
		fAlphaMult *= *pSwoosh->pfConstantAlpha;

	/* rebuild vertex buffer-- add points until we run out of valid history points or out of vertex buffer vertices */

	BoundingBox__Clear(&pSwoosh->WorldBoundingBox);	// rebuild as we build the swoosh
	pSwoosh->u16Flags |= SWOOSH_BOUNDING_BOX_VALID;
	pSwoosh->pVtxBuffer->Lock();
	int nVertexIndex = 0;
	int nPointIndex = pSwoosh->nSwooshPointIndex;
	ts_SwooshPoint *pSwooshPoint = &pSwoosh->pSwooshPoints[nPointIndex];
	while (pSwooshPoint->fLifetime > 0.0f)
	{
		/* get t value for evaluating envelopes-- runs from 1.0 (point at start of lifetime) to 0.0 (end of lifetime) */

		float fT = pSwooshPoint->fLifetime / pSwoosh->fPointLifetime;
		float fOneMinusT = 1.0f - fT;	// for evaluating envelopes-- 0.0 is start of envelope, 1.0 is end

		/* size */

		float fMagnitude = pSwooshPoint->fLineHalfLength;
		if (pSwoosh->pSwooshSource)
			fMagnitude *= Swoosh__EvaluateEnvelope(pSwoosh->pSwooshSource, X_SIZE_ENVELOPE, fOneMinusT);
		else
			fMagnitude *= fT;

		/* color */

		if (pSwoosh->pSwooshSource)
		{
			if ((pSwoosh->pSwooshSource->u32Flags & SWOOSH_COLOR_MASK) == SWOOSH_COLOR_RGB)
			{
				/* separate r, g, b envelopes */

				SwooshColor.R = (u8) (255.0f * fColorMult * Swoosh__EvaluateEnvelope(pSwoosh->pSwooshSource, RED_ENVELOPE, fOneMinusT));
				SwooshColor.G = (u8) (255.0f * fColorMult * Swoosh__EvaluateEnvelope(pSwoosh->pSwooshSource, GREEN_ENVELOPE, fOneMinusT));
				SwooshColor.B = (u8) (255.0f * fColorMult * Swoosh__EvaluateEnvelope(pSwoosh->pSwooshSource, BLUE_ENVELOPE, fOneMinusT));
			}
			else
			{
				/* single envelope that interpolates between two colors-- green and blue envelopes are unused */

				Swoosh__InterpolateColor(pSwoosh->pSwooshSource, Swoosh__EvaluateEnvelope(pSwoosh->pSwooshSource, RED_ENVELOPE, fOneMinusT), SwooshColor);

				/* add in the color multiplier */

				SwooshColor.R = (u8) (fColorMult * (float) SwooshColor.R);
				SwooshColor.G = (u8) (fColorMult * (float) SwooshColor.G);
				SwooshColor.B = (u8) (fColorMult * (float) SwooshColor.B);
			}
			SwooshColor.A = (u8) (fAlphaMult * 255.0f * Swoosh__EvaluateEnvelope(pSwoosh->pSwooshSource, ALPHA_ENVELOPE, fOneMinusT));
		}
		else
		{
			/* simple swoosh, created without source created/exported by pigpen. color rgb is already loaded in SwooshColor */

			SwooshColor.A = (u8) (fAlphaMult * pSwoosh->fMaxAlpha * fT);
		}

		/* each point in the history buffer contributes two vertices to the triangle strip */

		Vector3 vEndpoint = pSwooshPoint->vPoint + (pSwooshPoint->vLineDir * fMagnitude);
		BoundingBox__Include(&pSwoosh->WorldBoundingBox, vEndpoint);
		pSwoosh->pVtxBuffer->Vertex(nVertexIndex) = vEndpoint;
		if (pSwoosh->u16Flags & SWOOSH_IS_BLUR)
			pSwoosh->pVtxBuffer->TexCoordv2(0, nVertexIndex) = Swoosh__WorldToUV(worldToScreen, vEndpoint);
		pSwoosh->pVtxBuffer->Diffuse(nVertexIndex) = VtxBuffer::cvtRGBA(SwooshColor);
		nVertexIndex++;

		/* maybe adjust edge 2 color/alpha */

		SwooshColor.R = (u8) ((float) SwooshColor.R * pSwoosh->fEdge2ColorMult);
		SwooshColor.G = (u8) ((float) SwooshColor.G * pSwoosh->fEdge2ColorMult);
		SwooshColor.B = (u8) ((float) SwooshColor.B * pSwoosh->fEdge2ColorMult);
		SwooshColor.A = (u8) ((float) SwooshColor.A * pSwoosh->fEdge2AlphaMult);

		/* second vertex */

		vEndpoint = pSwooshPoint->vPoint - (pSwooshPoint->vLineDir * fMagnitude);
		BoundingBox__Include(&pSwoosh->WorldBoundingBox, vEndpoint);
		pSwoosh->pVtxBuffer->Vertex(nVertexIndex) = vEndpoint;
		if (pSwoosh->u16Flags & SWOOSH_IS_BLUR)
			pSwoosh->pVtxBuffer->TexCoordv2(0, nVertexIndex) = Swoosh__WorldToUV(worldToScreen, vEndpoint);
		pSwoosh->pVtxBuffer->Diffuse(nVertexIndex) = VtxBuffer::cvtRGBA(SwooshColor);
		nVertexIndex++;

		/* to previous point in history buffer (to go backwards in time) */

		if (--nPointIndex < 0)
			nPointIndex = pSwoosh->nSwooshPointCount - 1;
		pSwooshPoint = &pSwoosh->pSwooshPoints[nPointIndex];

		/* if we reached our starting point, we're done */

		if (nPointIndex == pSwoosh->nSwooshPointIndex)
			break;
	}

	/* if less than 3 points, nothing to render */

	if (nVertexIndex < 3)
	{
		pSwoosh->u16Flags &= ~SWOOSH_VISIBLE;

		/* if we are letting a deactivated swoosh die gracefully, once it's gone it's gone */

		if ((pSwoosh->u16Flags & SWOOSH_ADD_NEW_POINT) == 0)
			pSwoosh->u16Flags &= ~SWOOSH_ACTIVE;
	}
	else
	{
		/* swoosh is visible */

		pSwoosh->u16Flags |= SWOOSH_VISIBLE;

		/* usually we want the swoosh to end in a sharp tail, so get rid of last (oldest) triangle by removing a point */

		if ((pSwoosh->u16Flags & SWOOSH_NO_SHARP_TAIL) == 0)
			nVertexIndex--;

		/* and fill up rest of vertex buffer with last point to make degenerate triangles */

		if (nVertexIndex < pSwoosh->nSwooshPointCount * 2)
		{
			Vector3 vLastPoint = pSwoosh->pVtxBuffer->Vertex(nVertexIndex - 1);
			while (nVertexIndex < pSwoosh->nSwooshPointCount * 2)
			{
				pSwoosh->pVtxBuffer->Vertex(nVertexIndex) = vLastPoint;
				pSwoosh->pVtxBuffer->Diffuse(nVertexIndex) = 0;	// black with transparent alpha-- invisible
				nVertexIndex++;
			}
		}
	}

	/* done with vertex buffer */

	pSwoosh->pVtxBuffer->UnLock();
}

/* calculate and add a new point to history buffer */

void Swoosh__AddPoint(ts_Swoosh *pSwoosh, float fFrameSeconds)
{
	/* which method? */

	Vector3 vEndpoints[2];
	switch(pSwoosh->u16Flags & SWOOSH_TYPE_MASK)
	{
	case SWOOSH_TYPE_BONE:
		if (pSwoosh->u16Flags & SWOOSH_BONE_SWAP)
		{
			/* swoosh edge order is important, may want to swap bone endpoints */

			vEndpoints[0] = pSwoosh->pCActor->GetBoneWorldEndPosition(pSwoosh->nBoneID);
			vEndpoints[1] = pSwoosh->pCActor->GetBoneWorldStartPosition(pSwoosh->nBoneID);
		}
		else
		{
			/* this is the normal case */

			vEndpoints[0] = pSwoosh->pCActor->GetBoneWorldStartPosition(pSwoosh->nBoneID);
			vEndpoints[1] = pSwoosh->pCActor->GetBoneWorldEndPosition(pSwoosh->nBoneID);
		}

		/* need to adjust? the bone start and end settings can make the swoosh cover only a small */
		/* portion of the bone, or a line longer than the bone, or can reverse the swoosh edges */

		if (pSwoosh->u16Flags & SWOOSH_BONE_ADJUST)
		{
			ASSERT((pSwoosh->u16Flags & SWOOSH_BONE_SWAP) == 0);	// swap handles the special case of start == 1.0 and end == 0.0
			Vector3 vBoneEndpoints[2];
			vBoneEndpoints[0] = vEndpoints[0];
			vBoneEndpoints[1] = vEndpoints[1];
			if (pSwoosh->fBoneStart != 0.0f)
				vEndpoints[0] = vBoneEndpoints[0] + ((vBoneEndpoints[1] - vBoneEndpoints[0]) * pSwoosh->fBoneStart);
			if (pSwoosh->fBoneEnd != 0.0f)
				vEndpoints[1] = vBoneEndpoints[0] + ((vBoneEndpoints[1] - vBoneEndpoints[0]) * pSwoosh->fBoneEnd);
		}
		break;
	case SWOOSH_TYPE_ENDPOINT:
		ASSERT(pSwoosh->pvEndpoints[0]);
		ASSERT(pSwoosh->pvEndpoints[1]);
		vEndpoints[0] = *pSwoosh->pvEndpoints[0];
		vEndpoints[1] = *pSwoosh->pvEndpoints[1];
		break;
	default:
		ASSERT(false);	// unknown type
		break;
	}

	/* and add it */

	Swoosh__AddPoint(pSwoosh, fFrameSeconds, vEndpoints);
}

/* add a new point to history buffer */

void Swoosh__AddPoint(ts_Swoosh *pSwoosh, float fFrameSeconds, Vector3 *vEndpoints)
{
	/* last, possibly-valid point is useful for possible interpolation */

	ts_SwooshPoint *pPreviousSwooshPoint = &pSwoosh->pSwooshPoints[pSwoosh->nSwooshPointIndex];

	/* update history buffer index. do this before adding point so that if SWOOSH_ADD_NEW_POINT gets turned off */
	/* we will still point to the last point added */

	pSwoosh->nSwooshPointIndex++;
	if (pSwoosh->nSwooshPointIndex >= pSwoosh->nSwooshPointCount)
		pSwoosh->nSwooshPointIndex = 0;

	/* get center point and delta */

	ts_SwooshPoint *pNewSwooshPoint = &pSwoosh->pSwooshPoints[pSwoosh->nSwooshPointIndex];
	pNewSwooshPoint->vPoint = 0.5f * (vEndpoints[0] + vEndpoints[1]);
	pNewSwooshPoint->vLineDir = vEndpoints[0] - pNewSwooshPoint->vPoint;
	pNewSwooshPoint->fLineHalfLength = pNewSwooshPoint->vLineDir.Length();

	/* normalize delta-- since we have already calculated the length, do it ourselves instead of going through Normalize */

	if (pNewSwooshPoint->fLineHalfLength)
		pNewSwooshPoint->vLineDir /= pNewSwooshPoint->fLineHalfLength;

	/* set lifetime for new point */

	pNewSwooshPoint->fLifetime = pSwoosh->fPointLifetime + fFrameSeconds;	// will have fFrameSeconds subtracted in first advance

	/* tangent is invalid-- not calculated yet */

	pNewSwooshPoint->vTangent.x( MAXFLOAT);

	/* maybe interpolate to get some nice curves-- compensates for large movement on bones, etc */

	Swoosh__Interpolate(pSwoosh, pPreviousSwooshPoint, pNewSwooshPoint);
}

/* check if we need to interpolate between old and new points to add some intermediate points to make the swoosh nice and smooth.
pPreviousSwooshPoint and pNewSwooshPoint are in the passed swoosh's history buffer. in fact, pNewSwooshPoint is the entry at the
current value of pSwoosh->nSwooshPointIndex */

void Swoosh__Interpolate(ts_Swoosh *pSwoosh, ts_SwooshPoint *pPreviousSwooshPoint, ts_SwooshPoint *pNewSwooshPoint)
{
	/* maybe no interpolation on this swoosh */

	if (pSwoosh->u16Flags & SWOOSH_NO_INTERMEDIATE_POINTS)
		return;
	ASSERT(pSwoosh->pBallisticPath == NULL);	// never interpolate ballistc swooshes-- they don't need it

	/* only interpolate if previous point is valid */

	if (pPreviousSwooshPoint->fLifetime <= 0.0f)
		return;

	/* distance between previous and new point controls how many intermediate points are created, but subject to LOD control. */
	/* min and max distances are in meters, values are hand-tuned to keep nice swooshes when fairly close to camera */

#define HIRES_MAX_SWOOSH_DISTANCE	0.1f	// meters, allowed max distance between swoosh points-- nice curves
#define LORES_MAX_SWOOSH_DISTANCE	5.0f	// meters, allowed max distance between swoosh points-- jagged swooshes
	float fMaxSwooshDistance = HIRES_MAX_SWOOSH_DISTANCE;
	if (pSwoosh->fDistanceToNearestCamera > HIRES_LOD_RANGE)
		fMaxSwooshDistance += (LORES_MAX_SWOOSH_DISTANCE * ((pSwoosh->fDistanceToNearestCamera - HIRES_LOD_RANGE) / (pSwoosh->fLODDistance - HIRES_LOD_RANGE)));

	/* do we need to interpolate? look at distance between new point and previous point */

	float fDistanceSquared = (pNewSwooshPoint->vPoint - pPreviousSwooshPoint->vPoint).LengthSquared();
	if (fDistanceSquared <= fMaxSwooshDistance * fMaxSwooshDistance)
		return;

	/* interpolate between previous and new point. the way this works is that we calculate a spline that goes through both */
	/* points. at each point the spline is perpendicular to the point's line, so that subsequent splines will be first-order */
	/* continuous. Swoosh__CalculateTangent returns a unit vector */

	Vector3 vPreviousTangent = SwooshPoint__CalculateTangent(pPreviousSwooshPoint, pNewSwooshPoint);
	Vector3 vNewTangent = SwooshPoint__CalculateTangent(pNewSwooshPoint, pPreviousSwooshPoint);

	/* the length of the tangents is governed by distance between points. pick a value that gives a nice curve-- somewhat */
	/* arbitrary */

	float fDistance = Math::Sqrt(fDistanceSquared);
	vPreviousTangent *= fDistance * 0.25f;
	vNewTangent *= fDistance * 0.25f;

	/* and the spline control points are just the endpoints plus the tangents */

	Vector3 vPreviousControlPoint = pPreviousSwooshPoint->vPoint + vPreviousTangent;
	Vector3 vNewControlPoint = pNewSwooshPoint->vPoint + vNewTangent;

	/* how many intermediate points will we need? */

	int nCount = (int) (fDistance / fMaxSwooshDistance);
	ASSERT(nCount > 0);	// since fDistance should be > fMaxSwooshDistance
	if (nCount > pSwoosh->nSwooshPointCount - 2)
	{
		ASSERT(false);	// this is bad! it means that we are using all available points for just two main points in history buffer!
		nCount = pSwoosh->nSwooshPointCount - 2;
	}
	float fDeltaT = 1.0f / (float) (nCount + 1);	// e.g. if 1 intermediate point, fDeltaT = 0.5f
	Vector3 vDelta = (pNewSwooshPoint->vLineDir - pPreviousSwooshPoint->vLineDir) * fDeltaT;
	float fLengthDelta = (pNewSwooshPoint->fLineHalfLength - pPreviousSwooshPoint->fLineHalfLength) * fDeltaT;
	float fLifetimeDelta = (pNewSwooshPoint->fLifetime - pPreviousSwooshPoint->fLifetime) * fDeltaT;

	/* save new point, since we'll be adding intermediate points into the history buffer */

	ts_SwooshPoint SaveNewSwooshPoint = *pNewSwooshPoint;

	/* first values */

	float fT = fDeltaT;
	Vector3 vLineDir = pPreviousSwooshPoint->vLineDir + vDelta;
	float fLineHalfLength = pPreviousSwooshPoint->fLineHalfLength + fLengthDelta;
	float fLifetime = pPreviousSwooshPoint->fLifetime + fLifetimeDelta;

	/* loop through, adding intermediate points */

	for (int i = 0; i < nCount; i++)
	{
		/* fill in intermediate point */

		ts_SwooshPoint *pSwooshPoint = &pSwoosh->pSwooshPoints[pSwoosh->nSwooshPointIndex];
		pSwooshPoint->vLineDir = vLineDir;
		pSwooshPoint->vLineDir.Normalize();	// vLineDir does a linear interpolation between two unit vectors, intermediate values are not unit vectors
		pSwooshPoint->fLineHalfLength = fLineHalfLength;
		pSwooshPoint->fLifetime = fLifetime;
		pSwooshPoint->vTangent.x( MAXFLOAT);	// not valid, but should never be needed

		/* evaluate spline to find new point. the parametric equation for a point on a spline is:

			  (1 - t)^3 * p0  +  3 * (1 - t)^2 * t * p1  +  3 * (1 - t) * t ^ 2 * p2  +  t^3 * p3

			where t runs from 0.0 to 1.0 */

		float fTSquared = fT * fT;
		float fTCubed = fTSquared * fT;
		float fOneMinusT = 1.0f - fT;
		float fOneMinusTSquared = fOneMinusT * fOneMinusT;
		float fOneMinusTCubed = fOneMinusTSquared * fOneMinusT;
		pSwooshPoint->vPoint = fOneMinusTCubed * pPreviousSwooshPoint->vPoint;
		pSwooshPoint->vPoint += 3.0f * fOneMinusTSquared * fT * vPreviousControlPoint;
		pSwooshPoint->vPoint += 3.0f * fOneMinusT * fTSquared * vNewControlPoint;
		pSwooshPoint->vPoint += fTCubed * SaveNewSwooshPoint.vPoint;

		/* to next values */

		fT += fDeltaT;
		vLineDir += vDelta;
		fLineHalfLength += fLengthDelta;
		fLifetime += fLifetimeDelta;

		/* update history buffer index */

		pSwoosh->nSwooshPointIndex++;
		if (pSwoosh->nSwooshPointIndex >= pSwoosh->nSwooshPointCount)
			pSwoosh->nSwooshPointIndex = 0;
	}

	/* and finally tack original new point onto history buffer */

	pSwoosh->pSwooshPoints[pSwoosh->nSwooshPointIndex] = SaveNewSwooshPoint;
}

/* re-calculate ballistic swoosh's path */

void Swoosh__CalculateBallisticPath(ts_Swoosh *pSwoosh, float fFrameSeconds)
{
	/* start at beginning of history buffer */

	pSwoosh->nSwooshPointIndex = 0;

	/* for a start, we'll just have a flat ribbon */

	Vector3 vVel = pSwoosh->pBallisticPath->GetStartVel();
	float fYAngle = Math::ArcTan2(vVel.x(), vVel.z());
	Vector3 vDelta;
	float sin,cos;
	Math::SinCos(fYAngle, sin, cos);
	vDelta.Set( cos * pSwoosh->fPathHalfWidth,
					0.0f,
					sin * -pSwoosh->fPathHalfWidth);

	/* step along ballistic path and generate all points in history buffer */

	Vector3 vEndpoints[2];
	float fSeconds = 0;
	while (pSwoosh->nSwooshPointIndex < pSwoosh->nSwooshPointCount - 1)	// automatically wraps, stop before that
	{
		Vector3 vPoint = pSwoosh->pBallisticPath->Evaluate(fSeconds);
		vEndpoints[0] = vPoint - vDelta;
		vEndpoints[1] = vPoint + vDelta;
		Swoosh__AddPoint(pSwoosh, fFrameSeconds, vEndpoints);
		fSeconds += pSwoosh->fPointInterval;	// move along path
	}
}

/* kill passed swoosh */

void Swoosh__Kill(ts_Swoosh *pSwoosh)
{
	/* free buffers */

	delete pSwoosh->pStripBuffer;
	pSwoosh->pStripBuffer = NULL;
	delete pSwoosh->pVtxBuffer;
	pSwoosh->pVtxBuffer = NULL;
	memFree(pSwoosh->pSwooshPoints);
	pSwoosh->pSwooshPoints = NULL;

	/* clear record */

	pSwoosh->Handle = INVALID_HANDLE;

	/* move from active to free list */

	SLinkList__RemoveLink(&SwooshMgr.SwooshActive, pSwoosh);
	SLinkList__AddLinkToHead(&SwooshMgr.SwooshFree, pSwoosh);
}

/***************** math support for interpolation ***********************************/

/* given two lines (defined by a point and a vector), calculate a tangent for use in interpolation. the tangent is used to define */
/* a spline between the two lines that passes through both reference points. this routine calculates the tangent for one of the lines. */
/* the tangent is perpendicular to the line, and points to the projection of the other line's reference point onto the plane through */
/* the first line's reference point. the goal is to get a spline that represents a nice curve between the two lines, then we can */
/* interpolate slong the spline to get intermediate center points for extra history points */

Vector3 SwooshPoint__CalculateTangent(ts_SwooshPoint *pPoint1, ts_SwooshPoint *pPoint2)
{
	/* if point already has a tangent pointing back towards previous history point, then new tangent has to be directly opposite-- */
	/* first order continuous */

	if (pPoint1->vTangent.x() != MAXFLOAT)
		return(pPoint1->vTangent * -1.0f);	// tangent is unit vector

	/* calculate tangent. we have a point and a vector that describe the first line. imagine a plane through the point, perpendicular */
	/* to the vector. then we can always project the other line's point onto this plane. a reasonable tangent is the one that points */
	/* at the projected point */

	/* the general formula for a plane is:

			ax + by + cz + d = 0

		where (a, b, c) is the normal n and (x, y, z) is a point p on the plane. another way of representing this is:

			d = -(p dot n)

		to find the distance from an arbitrary point to the plane, you plug it into the equation. so we have:

			distance = (p2 dot n) - (p dot n)

		or:

			distance = (p2 - p) dot n

		then the projection of the point onto the plane is just moving along the normal the calculated distance, or:

			projected = p2 - n * ((p2 - p) dot n)

		easy! */

	Vector3 vProjected = pPoint2->vPoint - (pPoint1->vLineDir * Vector3::Dot((pPoint2->vPoint - pPoint1->vPoint), pPoint1->vLineDir));
	pPoint1->vTangent = vProjected - pPoint1->vPoint;
	if (pPoint1->vTangent.SafeNormalize() == false)
	{
		/* projected point is same as first line's reference point, pick an arbitrary tangent perpendicular to plane normal. */
		/* any unit vector, as long as tangent dot normal is 0 */

		if ((pPoint1->vLineDir.x() == 0.0f) && (pPoint1->vLineDir.z() == 0.0f))
			pPoint1->vTangent.Set(1.0f, 0.0f, 0.0f);
		else
			pPoint1->vTangent.Set(-pPoint1->vLineDir.z(), 0.0f, -pPoint1->vLineDir.x());
	}

	/* note that we have filled in point 1's tangent for possible use next time. typically this is pointing back towards previous */
	/* point in history buffer, but it's possible for this to be a calculation for the first point in the history buffer, which */
	/* doesn't have a previous point, so never had a tangent calculated */

	return(pPoint1->vTangent);
}

/***************** manipulate individual swooshes from outside manager-- always by handle ********************/

void Swoosh__Activate(t_Handle SwooshHandle)
{
	/* point to swoosh, bail if invalid id */

	ASSERT(SwooshHandle != INVALID_HANDLE);
	ts_Swoosh *pSwoosh = SWOOSH_GET_RECORD(SwooshHandle);
	if (pSwoosh->Handle != SwooshHandle)
		return;

	/* bail if already active and actively adding points, and not suspended. if SWOOSH_ADD_NEW_POINT is clear, then */
	/* swoosh was deactivated but left to fade out-- have to consider that as deactivated. if SWOOSH_LOD_SUSPENDED */
	/* is set, then history buffer is not getting updated, and failing to clear the history buffer will potentially */
	/* leave a gap */

	if ((pSwoosh->u16Flags & (SWOOSH_ACTIVE | SWOOSH_ADD_NEW_POINT | SWOOSH_LOD_SUSPENDED)) == (SWOOSH_ACTIVE | SWOOSH_ADD_NEW_POINT))
		return;

	/* set flags correctly */

	pSwoosh->u16Flags |= SWOOSH_ACTIVE | SWOOSH_ADD_NEW_POINT;
	pSwoosh->u16Flags &= ~SWOOSH_LOD_SUSPENDED;

	/* swoosh history buffer is empty on activation */

	for (int i = 0; i < pSwoosh->nSwooshPointCount; i++)
		pSwoosh->pSwooshPoints[i].fLifetime = 0.0f;	// point is valid only if lifetime > 0.0f
}
void Swoosh__Deactivate(t_Handle SwooshHandle, bool bHideImmediately /*= false*/)
{
	/* point to swoosh, bail if invalid id */

	ASSERT(SwooshHandle != INVALID_HANDLE);
	ts_Swoosh *pSwoosh = SWOOSH_GET_RECORD(SwooshHandle);
	if (pSwoosh->Handle != SwooshHandle)
		return;

	/* if user wants to hide immediately, just mark as not-active. otherwise let existing points die off */

	if (bHideImmediately)
		pSwoosh->u16Flags &= ~SWOOSH_ACTIVE;
	else
		pSwoosh->u16Flags &= ~SWOOSH_ADD_NEW_POINT;
}

/* kill passed swoosh */

void Swoosh__Kill(t_Handle SwooshHandle)
{
	/* point to swoosh, bail if invalid id */

	ASSERT(SwooshHandle != INVALID_HANDLE);
	ts_Swoosh *pSwoosh = SWOOSH_GET_RECORD(SwooshHandle);
	if (pSwoosh->Handle != SwooshHandle)
		return;
	Swoosh__Kill(pSwoosh);
}

/* pass the swoosh a place where it can find a color multiplier, 0.0 to 1.0 range, that can be used to tie the color of
the swoosh to the collision color data */

void Swoosh__SetColorMult(t_Handle SwooshHandle, float *pfColorMult)
{
	/* point to swoosh, bail if invalid id */

	ASSERT(SwooshHandle != INVALID_HANDLE);
	ts_Swoosh *pSwoosh = SWOOSH_GET_RECORD(SwooshHandle);
	if (pSwoosh->Handle != SwooshHandle)
		return;
	pSwoosh->pfColorMult = pfColorMult;
}

/* set a floor value for the color multiplier */

void Swoosh__SetColorMultFloor(t_Handle SwooshHandle, float fColorMultFloor)
{
	/* point to swoosh, bail if invalid id */

	ASSERT(SwooshHandle != INVALID_HANDLE);
	ts_Swoosh *pSwoosh = SWOOSH_GET_RECORD(SwooshHandle);
	if (pSwoosh->Handle != SwooshHandle)
		return;
	ASSERT(fColorMultFloor <= 1.0f);
	pSwoosh->fColorMultFloor = fColorMultFloor;
}

/* set a constant alpha for this swoosh, supports swoosh inheriting constant alpha from actor that owns it */

void Swoosh__SetConstantAlpha(t_Handle SwooshHandle, float *pfConstantAlpha)
{
	/* point to swoosh, bail if invalid id */

	ASSERT(SwooshHandle != INVALID_HANDLE);
	ts_Swoosh *pSwoosh = SWOOSH_GET_RECORD(SwooshHandle);
	if (pSwoosh->Handle != SwooshHandle)
		return;
	pSwoosh->pfConstantAlpha = pfConstantAlpha;
}

/* allow caller to request a sharp tail or not */

void Swoosh__SetSharpTail(t_Handle SwooshHandle, bool bSharpTail)
{
	/* point to swoosh, bail if invalid id */

	ASSERT(SwooshHandle != INVALID_HANDLE);
	ts_Swoosh *pSwoosh = SWOOSH_GET_RECORD(SwooshHandle);
	if (pSwoosh->Handle != SwooshHandle)
		return;
	if (bSharpTail)
		pSwoosh->u16Flags &= ~SWOOSH_NO_SHARP_TAIL;
	else
		pSwoosh->u16Flags |= SWOOSH_NO_SHARP_TAIL;
}

/* return value of passed envelope at current t value. return value is 0.0 to 1.0 */

float Swoosh__EvaluateEnvelope(ts_SwooshSource *pSwooshSource, int nIndex, float fT)
{
	ts_ParticleEnvelope *pEnvelope = &pSwooshSource->Envelope[nIndex];
	float fY;
	float fOneMinusT;

	/* what kind of envelope? */

	switch(pEnvelope->u8Type)
	{
	case SPLINE_ENVELOPE:
		fOneMinusT = 1.0f - fT;
//yuch-- this is not correct-- it ignores the x-position of the control points. the issue is that the spline equation
// is parameterized in t, and we want an equation y = f(x) where x proceeds linearly along the time axis (fT value)
		fY = (fOneMinusT * fOneMinusT * fOneMinusT * pEnvelope->fSplinePoint0Y) +
			  (3.0f * fOneMinusT * fOneMinusT * fT * pEnvelope->SplinePoint1.y()) +
			  (3.0f * fOneMinusT * fT * fT * pEnvelope->SplinePoint2.y()) +
			  (fT * fT * fT * pEnvelope->fSplinePoint3Y);
		break;
	case ADSR_ENVELOPE:
		if (fT < pEnvelope->SplinePoint1.x())
			fY = pEnvelope->fSplinePoint0Y + ((pEnvelope->SplinePoint1.y() - pEnvelope->fSplinePoint0Y) * fT / pEnvelope->SplinePoint1.x());
		else if (fT < pEnvelope->SplinePoint2.x())
			fY = pEnvelope->SplinePoint1.y() + ((pEnvelope->SplinePoint2.y() - pEnvelope->SplinePoint1.y()) * (fT - pEnvelope->SplinePoint1.x()) / (pEnvelope->SplinePoint2.x() - pEnvelope->SplinePoint1.x()));
		else
			fY = pEnvelope->SplinePoint2.y() + ((pEnvelope->fSplinePoint3Y - pEnvelope->SplinePoint2.y()) * (fT - pEnvelope->SplinePoint2.x()) / (1.0f - pEnvelope->SplinePoint2.x()));
		break;
	case STRAIGHT_ENVELOPE:
		fY = pEnvelope->fSplinePoint0Y + ((pEnvelope->fSplinePoint3Y - pEnvelope->fSplinePoint0Y) * fT);
		break;
	case FLAT_ENVELOPE:
		return(pEnvelope->fSplinePoint0Y);
	default:
		ASSERT(FALSE);	// unknown envelope type! bad data???
		return(1.0f);
	}

	/* check against clip values */

	if (fY > pEnvelope->fCeilingY)
		return(pEnvelope->fCeilingY);
	if (fY < pEnvelope->fFloorY)
		return(pEnvelope->fFloorY);
	return(fY);
}

/* interpolate between two colors to find current color. 1.0 is all first color, 0.0 is all second color.
this supports the color pair color control-- a single envelope controls interpolation between a start
and an end color */

void Swoosh__InterpolateColor(ts_SwooshSource *pSwooshSource, float fValue, ts_bRGBA &Color)
{
	ts_bRGBA *pColor0 = &pSwooshSource->Color[0];
	ts_bRGBA *pColor1 = &pSwooshSource->Color[1];

	Color.R = pColor1->R + (int) (fValue * (float) (pColor0->R - pColor1->R));
	Color.G = pColor1->G + (int) (fValue * (float) (pColor0->G - pColor1->G));
	Color.B = pColor1->B + (int) (fValue * (float) (pColor0->B - pColor1->B));
}

/* return a randomized value of the initial value, fValue +/- 0..fRandomValue */

float SwooshMgr__fRandomValue(float fValue, int nRandomPctValue)
{
	ASSERT((nRandomPctValue <= 100) || (nRandomPctValue == RANDOM_CENTER_ON_ZERO));
	if (nRandomPctValue != 0)
	{
		float fMinusOneToOne = SwooshRandom.NegPos();
		if (nRandomPctValue == RANDOM_CENTER_ON_ZERO)
			fValue = fValue * fMinusOneToOne;	// result is -fValue to fValue
		else if (fValue >= 0.0f)
		{
			fValue += fValue * ((float) nRandomPctValue / 100.0f) * fMinusOneToOne;
			if (fValue <= 0.0f)
				fValue = 0.001f;
		}
		else
		{
			fValue += fValue * ((float) nRandomPctValue / 100.0f) * fMinusOneToOne;
			if (fValue >= 0.0f)
				fValue = -0.001f;
		}
	}
	return(fValue);
}

