///////////////////////////////////////////////////////////////////////////////
// Generic module to manage special FX
///////////////////////////////////////////////////////////////////////////////
#include "Effects/EffectsPCH.h"

#include "Effects/sfx/sfxsys.h"
#include "Effects/sfx/specialfx.h"

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
SpecialFX::SpecialFX(
SpecialFXSystem &sfx,
const char *name) :
m_sfx(sfx)
{
	SetName(name);
	m_born = 0.0f;
	m_ttl = m_die = -1.0f;
	m_sibling = NULL;
	next = prev = NULL;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
SpecialFX::SpecialFX(
SpecialFXSystem &sfx,
const char *name,
float ttl) :
m_sfx(sfx)
{
	SetName(name);
	m_ttl = ttl;
	m_born = 0.0f;
	m_die = -1.0f;
	m_sibling = NULL;
	next = prev = NULL;
}

///////////////////////////////////////////////////////////////////////////////
// Make sure all siblings go away too
///////////////////////////////////////////////////////////////////////////////
SpecialFX::~SpecialFX()
{
	// Delete the siblings
	delete m_sibling;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void SpecialFX::SetName(
const char *name)
{
	if (!name)
		m_name[0] = '\0';
	else
	{
		ASSERT(strlen(name) < sizeof(m_name));
		strcpy(m_name, name);
	}
}

///////////////////////////////////////////////////////////////////////////////
// Reset the TTL
///////////////////////////////////////////////////////////////////////////////
void SpecialFX::ResetTTL(
float newTTL,
bool includeSiblings)
{
	m_ttl = newTTL;
	if (m_ttl > 0.0f)
		m_die = g_timer.GetEndSec() + m_ttl;
	else
		m_die = -1.0f;

	if (includeSiblings && m_sibling)
		m_sibling->ResetTTL(newTTL, includeSiblings);
}

///////////////////////////////////////////////////////////////////////////////
// Add a sibling on this guy (deleted when destructor is called)
///////////////////////////////////////////////////////////////////////////////
void SpecialFX::AddSibling(
SpecialFX *sibling)
{
	if (!sibling)
		return;

	// Hook it into the list
	sibling->m_sibling = m_sibling;
	m_sibling = sibling;
	m_sibling->OnAdd(g_timer.GetEndSec());
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void SpecialFX::OnAdd(float birth)
{
	m_born = birth;
	if (m_ttl > 0.0f)
		m_die = birth + m_ttl;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
bool SpecialFX::TimeToDie(float now)
{
	// Do we need to get out
	if (!TTLBased())
		return false;
	else // Is it time to die?
		return (now >= m_die);
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void SpecialFX::Update(
float now,
float deltaT)
{
	// Update our siblings
	while (m_sibling)
	{
		// Time to die?
		if (!m_sibling->TimeToDie(now))
		{
			m_sibling->Update(now, deltaT);
			break; // get out! (the above update updated everyone else)
		}
		else
		{
			// Unlink the SLL
			SpecialFX *killMe = m_sibling;
			m_sibling = m_sibling->m_sibling;
			killMe->m_sibling = NULL;

			// Going to kill
			killMe->OnRemove();
			delete killMe;
		}
	}
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void SpecialFXSticky::MoveTo(const Vector3 &pos)
{
	m_sticky.SetInWorld(NULL, pos);
	SpecialFX::MoveTo(pos);
}


