///////////////////////////////////////////////////////////////////////////////
// The camera group allows a bunch of cameras to be linked together in a
// somewhat meaningful way.
///////////////////////////////////////////////////////////////////////////////
#include "camera/CameraPCH.h"

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
CameraGroup::CameraGroup(
const char *name,
bool loop) :
m_list(true)
{
	// copy in the name
	strncpy(m_name, name, sizeof(m_name) - 1);
	m_name[sizeof(m_name) - 1] = '\0';

	// We get this once bound to a scene
	m_scene = NULL;

	// No camera is selected yet
	m_selected = NULL;
	m_lock = 0;

	// Do they want to loop?
	Loop(loop);
}

///////////////////////////////////////////////////////////////////////////////
// Add a camera to the group
///////////////////////////////////////////////////////////////////////////////
void CameraGroup::AddHead(
Camera *camera)
{
	if (!camera)
		return;
	camera->SetGroup(this);
	m_list.AddHead(camera);
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void CameraGroup::AddTail(
Camera *camera)
{
	if (!camera)
		return;
	camera->SetGroup(this);
	m_list.AddTail(camera);
}

///////////////////////////////////////////////////////////////////////////////
// Remove a camera from the group
///////////////////////////////////////////////////////////////////////////////
void CameraGroup::Remove(
Camera *camera)
{
	if (camera)
	{
		camera->SetGroup(NULL);
		m_list.Remove(camera);
	}
}

///////////////////////////////////////////////////////////////////////////////
// Get the first/last camera
///////////////////////////////////////////////////////////////////////////////
Camera *CameraGroup::First()
{
	return m_list.Head();
}

///////////////////////////////////////////////////////////////////////////////
// Get the first/last camera
///////////////////////////////////////////////////////////////////////////////
Camera *CameraGroup::Last()
{
	return m_list.Tail();
}

///////////////////////////////////////////////////////////////////////////////
// Select a the previous/next camera
///////////////////////////////////////////////////////////////////////////////
Camera *CameraGroup::Next(
Camera *camera)
{
	if (!camera)
		return NULL;

	Camera *next = camera->next;
	if (m_loop && !next)
		next = m_list.Head();

	return next;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
Camera *CameraGroup::Previous(
Camera *camera)
{
	if (!camera)
		return NULL;

	Camera *prev = camera->prev;
	if (m_loop && !prev)
		prev = m_list.Tail();

	return prev;
}

///////////////////////////////////////////////////////////////////////////////
// Find a camera
///////////////////////////////////////////////////////////////////////////////
Camera *CameraGroup::Find(const char *name)
{
	Camera *camera = m_list.Head();
	while (camera)
	{
		if (strcmpi(camera->Name(), name) == 0)
			return camera;

		camera = camera->next;
	}

	return NULL;
}

///////////////////////////////////////////////////////////////////////////////
// Select a camera for this group
///////////////////////////////////////////////////////////////////////////////
bool CameraGroup::Select(
Camera *camera)
{
	if (!camera)
		return false;

	// Are we locked?
	if (Locked())
		return false;

	if (camera == m_selected)
		return true;

	Camera *lastCamera = Deactivate();
	m_selected = camera;
	Activate(lastCamera);
	
	return true;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
bool CameraGroup::Select(
const char *cameraName)
{
	return Select(Find(cameraName));
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
Camera &CameraGroup::Selected()
{
	ASSERT(m_selected);
	return *m_selected;
}

///////////////////////////////////////////////////////////////////////////////
// How many cameras are there
///////////////////////////////////////////////////////////////////////////////
int CameraGroup::Cameras()
{
	return m_list.Items();
}

///////////////////////////////////////////////////////////////////////////////
// Lock the selection
///////////////////////////////////////////////////////////////////////////////
bool CameraGroup::Lock(
bool lock)
{
	if (lock)
		++m_lock;
	else
		--m_lock;

	ASSERT(m_lock >= 0);
	return Locked();
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
bool CameraGroup::Locked()
{
	return (m_lock > 0);
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void CameraGroup::Update()
{
}

///////////////////////////////////////////////////////////////////////////////
// This is to ensure the camera has the scene
///////////////////////////////////////////////////////////////////////////////
void CameraGroup::BindScene(CameraScene &scene)
{
	m_scene = &scene;
}

///////////////////////////////////////////////////////////////////////////////
// These are called when we gain/lose focus on the stack
///////////////////////////////////////////////////////////////////////////////
void CameraGroup::Activate(
Camera *lastCamera)
{
	ASSERT(m_selected);

	// Make sure the scene is bound
	ASSERT(m_scene);
	m_selected->BindScene(*m_scene);

	// Inherit the settings
	if (lastCamera)
		*m_selected = *lastCamera;

	// Activate it
	m_selected->Activate();
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
Camera *CameraGroup::Deactivate()
{
	if (m_selected)
		m_selected->Deactivate();
	return m_selected;
}


