///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
#include "camera/CameraPCH.h"

// The singleton
CameraDirector *g_cameraDirectorPtr = NULL;

///////////////////////////////////////////////////////////////////////////////
// Intialize creates us
///////////////////////////////////////////////////////////////////////////////
CameraDirector::CameraDirector() :
m_sceneList(false),
m_scenePool(8, 4)
{
	// No default scene yet
	m_defaultScene = NULL;
}

///////////////////////////////////////////////////////////////////////////////
// cleanup all of our lists
///////////////////////////////////////////////////////////////////////////////
CameraDirector::~CameraDirector()
{
	// Clean up stuff
	CameraScene *scene = m_sceneList.Tail();
	while (scene)
	{
		DestroyScene(scene);
		scene = m_sceneList.Tail();
	}
}

///////////////////////////////////////////////////////////////////////////////
// Initialize everything
///////////////////////////////////////////////////////////////////////////////
void CameraDirector::Initialize()
{
	// Make sure we are cool
	if(g_cameraDirectorPtr == NULL)
	{
		// Create the singleton
		g_cameraDirectorPtr = new CameraDirector();
	}
	ASSERT(g_cameraDirectorPtr);
}

///////////////////////////////////////////////////////////////////////////////
// Shut it down
///////////////////////////////////////////////////////////////////////////////
void CameraDirector::Terminate()
{
	delete g_cameraDirectorPtr;
	g_cameraDirectorPtr = NULL;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
CameraScene *CameraDirector::CreateScene(
Texture *realtimeTexture)
{
	// Make the buffer
	void *buffer = m_scenePool.NewPtr();
	if (!buffer)
		return NULL;

	// Create it
	CameraScene *scene = new (buffer) CameraScene(realtimeTexture);

	// Did it work?
	if (!scene->Scene())
	{
		m_scenePool.Delete(scene);
		return NULL;
	}

	// Add it to the list
	m_sceneList.AddTail(scene);

	// Is there a default scene?
	if (!m_defaultScene)
		m_defaultScene = scene;

	// Give them back the scene
	return scene;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void CameraDirector::DestroyScene(
CameraScene *scene)
{
	// Get the scene
	scene = UseScene(scene);
	if (!scene)
		return;

	// Unlink it from the list
	m_sceneList.Unlink(scene);

	// Return it to the pool
	m_scenePool.Delete(scene);

	// Uh-oh can't be the default
	if (m_defaultScene == scene)
		m_defaultScene = NULL;
}

///////////////////////////////////////////////////////////////////////////////
// Set the default scene
///////////////////////////////////////////////////////////////////////////////
void CameraDirector::SetDefaultScene(
CameraScene *scene)
{
	m_defaultScene = scene;
}

///////////////////////////////////////////////////////////////////////////////
// Get the default scene
///////////////////////////////////////////////////////////////////////////////
CameraScene *CameraDirector::GetDefaultScene()
{
	return m_defaultScene;
}

///////////////////////////////////////////////////////////////////////////////
// Access the camera matrix
///////////////////////////////////////////////////////////////////////////////
Cartesian4x4 &CameraDirector::GetCameraMatrix(
CameraScene *scene)
{
	scene = UseScene(scene);

	ASSERT(scene);
	return scene->CameraMatrix();
}

///////////////////////////////////////////////////////////////////////////////
// Set the camera for a scene
///////////////////////////////////////////////////////////////////////////////
void CameraDirector::PushGroup(
CameraGroup *cameraGroup,
CameraScene *scene)
{
	ASSERT(cameraGroup != NULL);

	// Get the scene
	scene = UseScene(scene);
	if (!scene)
		return;

	// Set the camera as active for this scene
	scene->PushGroup(cameraGroup);
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void CameraDirector::PushGroup(
const char *groupName,
CameraScene *scene)
{
	ASSERT(groupName != NULL);

	// Get the scene
	scene = UseScene(scene);
	if (!scene)
		return;

	// Set the camera as active for this scene
	scene->PushGroup(groupName);
}

///////////////////////////////////////////////////////////////////////////////
// Pop the camera for the scene
///////////////////////////////////////////////////////////////////////////////
CameraGroup *CameraDirector::PopGroup(
CameraGroup *cameraGroup,
CameraScene *scene)
{
	// Get the scene
	scene = UseScene(scene);
	if (!scene)
		return NULL;

	// Give them back the camera from the top
	return scene->PopGroup(cameraGroup);
}

///////////////////////////////////////////////////////////////////////////////
// Pop the camera for the scene
///////////////////////////////////////////////////////////////////////////////
CameraGroup *CameraDirector::PopGroup(
const char *groupName,
CameraScene *scene)
{
	// Get the scene
	scene = UseScene(scene);
	if (!scene)
		return NULL;

	// Give them back the camera from the top
	return scene->PopGroup(groupName);
}

///////////////////////////////////////////////////////////////////////////////
// What is the current camera for the scene
///////////////////////////////////////////////////////////////////////////////
Camera &CameraDirector::CurrentCamera(
CameraScene *scene)
{
	// Get the scene
	scene = UseScene(scene);
	ASSERT(scene);

	// Give them back the camera from the top
	return scene->CurrentCamera();
}

///////////////////////////////////////////////////////////////////////////////
// What is the current camera for the scene
///////////////////////////////////////////////////////////////////////////////
bool CameraDirector::HasRealCamera(
CameraScene *scene)
{
	// Get the scene
	scene = UseScene(scene);
	ASSERT(scene);

	// Give them back the camera from the top
	return scene->HasRealCamera();
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
ProjectionController &CameraDirector::GetCurrentProjection(
CameraScene *scene)
{
	// Get the scene
	scene = UseScene(scene);
	ASSERT(scene);

	return scene->GetCurrentProjection();
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
ViewportController &CameraDirector::GetCurrentViewport(
CameraScene *scene)
{
	// Get the scene
	scene = UseScene(scene);
	ASSERT(scene);

	return scene->GetCurrentViewport();
}

///////////////////////////////////////////////////////////////////////////////
// Setup the Master projection and scene controllers (set to NULL to go
// back to the default.
///////////////////////////////////////////////////////////////////////////////
void CameraDirector::SetMasterProjection(
ProjectionController *controller,
CameraScene *scene)
{
	// Get the scene
	scene = UseScene(scene);
	if (!scene)
		return;

	// Set it
	scene->SetMasterProjection(controller);
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
ProjectionController &CameraDirector::GetMasterProjection(
CameraScene *scene)
{
	// Get the scene
	scene = UseScene(scene);
	ASSERT(scene);
	return scene->MasterProjection();
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void CameraDirector::SetMasterViewport(
ViewportController *controller,
CameraScene *scene)
{
	// Get the scene
	scene = UseScene(scene);
	if (!scene)
		return;

	// Set it
	scene->SetMasterViewport(controller);
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
ViewportController &CameraDirector::GetMasterViewport(
CameraScene *scene)
{
	// Get the scene
	scene = UseScene(scene);
	ASSERT(scene);
	return scene->MasterViewport();
}

///////////////////////////////////////////////////////////////////////////////
// Update the camera
///////////////////////////////////////////////////////////////////////////////
void CameraDirector::Update()
{
	CameraScene *scene = m_sceneList.Head();
	while (scene)
	{
		// work
		scene->Update();

		// next!
		scene = scene->next;
	}
}

///////////////////////////////////////////////////////////////////////////////
// Render right after the main render
///////////////////////////////////////////////////////////////////////////////
void CameraDirector::PostRenderUpdate()
{
	CameraScene *scene = m_sceneList.Head();
	while (scene)
	{
		// Do the work
		scene->PostRenderUpdate();

		// next!
		scene = scene->next;
	}
}

///////////////////////////////////////////////////////////////////////////////
// Camera group support (any group you give to the director
// will be deleted on destruction)
///////////////////////////////////////////////////////////////////////////////
void CameraDirector::AddGroup(
CameraGroup *group,
CameraScene *scene)
{
	// Get the scene
	scene = UseScene(scene);
	if (scene)
		scene->AddGroup(group);
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void CameraDirector::RemoveGroup(
CameraGroup *group,
CameraScene *scene)
{
	// Get the scene
	scene = UseScene(scene);
	if (scene)
		scene->RemoveGroup(group);
}

///////////////////////////////////////////////////////////////////////////////
// Find a group
///////////////////////////////////////////////////////////////////////////////
CameraGroup *CameraDirector::FindGroup(
const char *groupName,
CameraScene *scene)
{
	// Get the scene
	scene = UseScene(scene);
	if (!scene)
		return NULL;

	return scene->FindGroup(groupName);
}

///////////////////////////////////////////////////////////////////////////////
// This is the majik call
///////////////////////////////////////////////////////////////////////////////
bool CameraDirector::SelectCamera(
const char *groupName,
const char *cameraName,
CameraScene *scene)
{
// Get the scene
	scene = UseScene(scene);
	if (!scene)
		return false;

	return scene->SelectCamera(groupName, cameraName);
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
CameraScene *CameraDirector::UseScene(
CameraScene *scene)
{
	if (scene)
		return scene;
	else if (m_defaultScene)
		return m_defaultScene;
	else
		return NULL;
}

