/***************************************************************************/
// A super-nice Vector class that is fast too.
/***************************************************************************/
#include "Math/MathClass.h"

/***************************************************************************/
// assignment constructor
/***************************************************************************/
inline Vector2::Vector2(
float x,
float y)
{
	this->x = x;
	this->y = y;
}

/***************************************************************************/
/***************************************************************************/
inline Vector2::Vector2(
const float *array)
{
	x = array[0];
	y = array[1];
}

/***************************************************************************/
/***************************************************************************/
inline Vector2::Vector2(
const ts_Vct2f &v)
{
	x = v.X;
	y = v.Y;
}

/***************************************************************************/
// Clear to 0
/***************************************************************************/
inline void Vector2::Clear(void)
{
	x = y = 0.0f;
}

/***************************************************************************/
// Set the values
/***************************************************************************/
inline void Vector2::Set(
float x,
float y)
{
	this->x = x;
	this->y = y;
}

/***************************************************************************/
// Set the values
/***************************************************************************/
inline void Vector2::Set(
const float *xyArray)
{
	x = xyArray[0];
	y = xyArray[1];
}

/***************************************************************************/
// Return this Dot v
/***************************************************************************/
inline float Vector2::Dot(
const Vector2 &v) const
{
	return(x * v.x + y * v.y);
}

/***************************************************************************/
// Return [this->x * v.x, this->y * v.y]
/***************************************************************************/
inline Vector2 Vector2::Diag(
const Vector2 &v) const
{
	Vector2 temp;
	temp.x = x * v.x;
	temp.y = y * v.y;
	return(temp);
}

/***************************************************************************/
// Return the Length of this Vector2
/***************************************************************************/
inline float Vector2::Length(void) const
{
	return(Math::Sqrt(x * x + y * y));
}

/***************************************************************************/
// Return the Length squared of this vector
/***************************************************************************/
inline float Vector2::LengthSquared(void) const
{
	return(x * x + y * y);
}

/***************************************************************************/
// Normalize the Vector2
/***************************************************************************/
inline void Vector2::Normalize(void)
{
	float invLength = 1.0f / Math::Sqrt(x * x + y * y);
	x *= invLength;
	y *= invLength;
}

/***************************************************************************/
// Return the normalized Vector
/***************************************************************************/
inline Vector2 Vector2::Normalized(void) const
{
	float invLength = 1.0f / Math::Sqrt(x * x + y * y);
	Vector2 n;
	n.x = x * invLength;
	n.y = y * invLength;
	return(n);
}

/***************************************************************************/
// Negate the Vector2
/***************************************************************************/
inline void Vector2::Negate(void)
{
	x = -x;
	y = -y;
}

/***************************************************************************/
// Return v1 Dot v2
/***************************************************************************/
inline float Vector2::Dot(
const Vector2 &v1,
const Vector2 &v2)
{
	return(v1.x * v2.x + v1.y * v2.y);
}

/***************************************************************************/
// Return [v1.x * v2.x, v1.y * v2.y]
/***************************************************************************/
inline Vector2 Vector2::Diag(
const Vector2 &v1,
const Vector2 &v2)
{
	Vector2 temp;
	temp.x = v1.x * v2.x;
	temp.y = v1.y * v2.y;
	return(temp);
}

/***************************************************************************/
// Return the Length of this Vector2
/***************************************************************************/
inline float Vector2::Length(
const Vector2 &v)
{
	return(Math::Sqrt(v.x * v.x + v.y * v.y));
}

/***************************************************************************/
// Normalize the Vector2
/***************************************************************************/
inline void Vector2::Normalize(
Vector2 &v)
{
	float invLength = 1.0f / Math::Sqrt(v.x * v.x + v.y * v.y);
	v.x *= invLength;
	v.y *= invLength;
}

/***************************************************************************/
// Equality
/***************************************************************************/
inline int Vector2::operator==(
const Vector2 &v) const
{
	return(x == v.x && y == v.y);
}

/***************************************************************************/
// Non-equality
/***************************************************************************/
inline int Vector2::operator!=(
const Vector2 &v) const
{
	return(x != v.x || y != v.y);
}

/***************************************************************************/
// Assignment
/***************************************************************************/
inline Vector2 &Vector2::operator=(
const Vector2 &v)
{
	x = v.x;
	y = v.y;
	return(*this);
}

/***************************************************************************/
// Assignment
/***************************************************************************/
inline Vector2 &Vector2::operator=(
const float f)
{
	x = y = f;
	return(*this);
}

/***************************************************************************/
// Assignment
/***************************************************************************/
inline Vector2 &Vector2::operator=(
int i)
{
	x = y = (float)i;
	return(*this);
}

/***************************************************************************/
// Addition
/***************************************************************************/
inline Vector2 &Vector2::operator+=(
const Vector2 &v)
{
	x += v.x;
	y += v.y;

	return(*this);
}

/***************************************************************************/
// Subtraction
/***************************************************************************/
inline Vector2 &Vector2::operator-=(
const Vector2 &v)
{
	x -= v.x;
	y -= v.y;

	return(*this);
}

/***************************************************************************/
// Scaling
/***************************************************************************/
inline Vector2 &Vector2::operator*=(
float scale)
{
	x *= scale;
	y *= scale;

	return(*this);
}

/***************************************************************************/
// Scaling
/***************************************************************************/
inline Vector2 &Vector2::operator/=(
float scale)
{
	float invScale = 1.0f / scale;
	x *= invScale;
	y *= invScale;

	return(*this);
}

/***************************************************************************/
// Addition
/***************************************************************************/
inline Vector2 Vector2::operator+(
const Vector2 &v) const
{
	Vector2 temp;
	temp.x = x + v.x;
	temp.y = y + v.y;
	return(temp);
}

/***************************************************************************/
// Subtraction
/***************************************************************************/
inline Vector2 Vector2::operator-(
const Vector2 &v) const
{
	Vector2 temp;
	temp.x = x - v.x;
	temp.y = y - v.y;
	return(temp);
}

/***************************************************************************/
// Scaling
/***************************************************************************/
inline Vector2 Vector2::operator*(
float scale) const
{
	Vector2 temp;
	temp.x = x * scale;
	temp.y = y * scale;
	return(temp);
}

/***************************************************************************/
// Scaling
/***************************************************************************/
inline Vector2 Vector2::operator/(
float scale) const
{
	float invScale = 1.0f / scale;
	Vector2 temp;
	temp.x = x * invScale;
	temp.y = y * invScale;
	return(temp);
}

/***************************************************************************/
// Negation
/***************************************************************************/
inline Vector2 Vector2::operator-(
void) const
{
	Vector2 temp;
	temp.x = -x;
	temp.y = -y;
	return(temp);
}

/***************************************************************************/
// Get at the Vector2 like an array of reals
/***************************************************************************/
inline float &Vector2::operator[](
int x0y1)
{
	return(((float *)&x)[x0y1]);
}

/***************************************************************************/
// Casts
/***************************************************************************/
inline Vector2::operator ts_Vct2f*(void) const
{
	return((ts_Vct2f *)&x);
}

/***************************************************************************/
/***************************************************************************/
inline Vector2::operator ts_Vct2f(void) const
{
	return(*((ts_Vct2f *)&x));
}

/***************************************************************************/
/***************************************************************************/
inline Vector2::operator float*(void)
{
	return(&x);
}

/***************************************************************************/
// assignment constructor
/***************************************************************************/
inline Vector3::Vector3(
float x,
float y,
float z)
{
	this->x = x;
	this->y = y;
	this->z = z;
}

/***************************************************************************/
// Constructor for a Vector2
/***************************************************************************/
inline Vector3::Vector3(
const Vector2 &v)
{
	x = v.x;
	y = v.y;
	z = 0.0f;
}

/***************************************************************************/
/***************************************************************************/
inline Vector3::Vector3(
const float *array)
{
	x = array[0];
	y = array[1];
	z = array[2];
}

/***************************************************************************/
/***************************************************************************/
inline Vector3::Vector3(
const ts_Vct2f &v)
{
	x = v.X;
	y = v.Y;
	z = 0.0f;
}

/***************************************************************************/
/***************************************************************************/
inline Vector3::Vector3(
const ts_Vct3f &v)
{
	x = v.X;
	y = v.Y;
	z = v.Z;
}

/***************************************************************************/
/***************************************************************************/
inline Vector3::Vector3(
const ts_Vct4f &v)
{
	x = v.X;
	y = v.Y;
	z = v.Z;
}

/***************************************************************************/
// Clear to 0
/***************************************************************************/
inline void Vector3::Clear(void)
{
	x = y = z = 0.0f;
}

/***************************************************************************/
// Set the values
/***************************************************************************/
inline void Vector3::Set(
float x,
float y,
float z)
{
	this->x = x;
	this->y = y;
	this->z = z;
}

/***************************************************************************/
// Set the values
/***************************************************************************/
inline void Vector3::Set(
const float *xyzArray)
{
	x = xyzArray[0];
	y = xyzArray[1];
	z = xyzArray[2];
}

/***************************************************************************/
// Set the values
/***************************************************************************/
inline void Vector3::Set(
const ts_Vct3f &v)
{
	x = v.X;
	y = v.Y;
	z = v.Z;
}

/***************************************************************************/
// Get at it
/***************************************************************************/
inline void Vector3::Get3(
ts_Vct3f &v) const
{
	v.X = x;
	v.Y = y;
	v.Z = z;
}

/***************************************************************************/
// Get at it
/***************************************************************************/
inline void Vector3::Get4(
ts_Vct4f &v) const
{
	v.X = x;
	v.Y = y;
	v.Z = z;
	v.W = 1.0f;
}

/***************************************************************************/
// Return this Dot v
/***************************************************************************/
inline float Vector3::Dot(
Vector3CRef v) const
{
	return(x * v.x + y * v.y + z * v.z);
}

/***************************************************************************/
// Return this X v
/***************************************************************************/
inline Vector3 Vector3::Cross(
Vector3CRef v) const
{
	Vector3 temp;
	temp.x = y * v.z - v.y * z;
	temp.y = -(x * v.z - v.x * z);
	temp.z = x * v.y - v.x * y;
	return(temp);
}

/***************************************************************************/
// Return [this->x * v.x, this->y * v.y, this->z * v.z]
/***************************************************************************/
inline Vector3 Vector3::Diag(
Vector3CRef v) const
{
	Vector3 temp;
	temp.x = x * v.x;
	temp.y = y * v.y;
	temp.z = z * v.z;
	return(temp);
}

/***************************************************************************/
// Find the projection of v onto this
/***************************************************************************/
inline Vector3 Vector3::Projection(Vector3CRef v)
{
	return(*this * Dot(v) / LengthSquared());
}

/***************************************************************************/
// Return the Length of this Vector3
/***************************************************************************/
inline float Vector3::Length(void) const
{
	return(Math::Sqrt(x * x + y * y + z * z));
}

/***************************************************************************/
// Return the Length of this Vector3's x & z components
/***************************************************************************/
inline float Vector3::LengthXZ(void) const
{
	return(Math::Sqrt(x * x + z * z));
}


/***************************************************************************/
// Return the Length squared of this vector
/***************************************************************************/
inline float Vector3::LengthSquared(void) const
{
	return(x * x + y * y + z * z);
}

/***************************************************************************/
// Return the Length squared of this vector
/***************************************************************************/
inline float Vector3::LengthXZSquared(void) const
{
	return(x * x +  z * z);
}

/***************************************************************************/
// Normalize the Vector3
/***************************************************************************/
inline void Vector3::Normalize(void)
{
	float invLength = 1.0f / Math::Sqrt(x * x + y * y + z * z);
	x *= invLength;
	y *= invLength;
	z *= invLength;
}

/***************************************************************************/
// Return the normalized Vector
/***************************************************************************/
inline Vector3 Vector3::Normalized(void) const
{
	float invLength = 1.0f / Math::Sqrt(x * x + y * y + z * z);
	Vector3 n;
	n.x = x * invLength;
	n.y = y * invLength;
	n.z = z * invLength;
	return(n);
}

/***************************************************************************/
// Negate the Vector3
/***************************************************************************/
inline void Vector3::Negate(void)
{
	x = -x;
	y = -y;
	z = -z;
}

/***************************************************************************/
// Negate the Vector3
/***************************************************************************/
inline Vector3 Vector3::Negated(void) const
{
	Vector3 n;
	n.x = -x;
	n.y = -y;
	n.z = -z;
	return(n);
}

/***************************************************************************/
// Return v1 Dot v2
/***************************************************************************/
inline float Vector3::Dot(
Vector3CRef v1,
Vector3CRef v2)
{
	return(v1.x * v2.x + v1.y * v2.y + v1.z * v2.z);
}

/***************************************************************************/
// Return v1 X v2
/***************************************************************************/
inline Vector3 Vector3::Cross(
Vector3CRef v1,
Vector3CRef v2)
{
	Vector3 temp;
	temp.x = v1.y * v2.z - v2.y * v1.z;
	temp.y = -(v1.x * v2.z - v2.x * v1.z);
	temp.z = v1.x * v2.y - v2.x * v1.y;
	return(temp);
}

/***************************************************************************/
// Return [v1.x * v2.x, v1.y * v2.y, v1.z * v2.z]
/***************************************************************************/
inline Vector3 Vector3::Diag(
Vector3CRef v1,
Vector3CRef v2)
{
	Vector3 temp;
	temp.x = v1.x * v2.x;
	temp.y = v1.y * v2.y;
	temp.z = v1.z * v2.z;
	return(temp);
}

/***************************************************************************/
// Return the Length of this Vector3
/***************************************************************************/
inline float Vector3::Length(
Vector3CRef v)
{
	return(Math::Sqrt(v.x * v.x + v.y * v.y + v.z * v.z));
}

/***************************************************************************/
// Return the Length of this Vector3's x & z components
/***************************************************************************/
inline float Vector3::LengthXZ(
Vector3CRef v)
{
	return(Math::Sqrt(v.x * v.x + v.z * v.z));
}


//***************************************************************************/
// Return the Length of this Vector3
/***************************************************************************/
inline float Vector3::LengthSquared(
Vector3CRef v)
{
	return(v.x * v.x + v.y * v.y + v.z * v.z);
}

/***************************************************************************/
// Return the Length of this Vector3's x & z components
/***************************************************************************/
inline float Vector3::LengthXZSquared(
Vector3CRef v)
{
	return(v.x * v.x + v.z * v.z);
}

/***************************************************************************/
// Normalize the Vector3
/***************************************************************************/
inline void Vector3::Normalize(
Vector3 &v)
{
	float invLength = 1.0f / Math::Sqrt(v.x * v.x + v.y * v.y + v.z * v.z);
	v.x *= invLength;
	v.y *= invLength;
	v.z *= invLength;
}

/***************************************************************************/
// Find the projection of b onto a
/***************************************************************************/
inline Vector3 Vector3::Projection(
const Vector3& a,
const Vector3& b)
{
	return(a * Vector3::Dot(a, b) / Vector3::Dot(a, a));
}

/***************************************************************************/
// Equality
/***************************************************************************/
inline int Vector3::operator==(
Vector3CRef v) const
{
	return(x == v.x && y == v.y && z == v.z);
}

/***************************************************************************/
// Equality
/***************************************************************************/
inline int Vector3::operator==(
const ts_Vct3f &v) const
{
	return(x == v.X && y == v.Y && z == v.Z);
}

/***************************************************************************/
// Non-equality
/***************************************************************************/
inline int Vector3::operator!=(
Vector3CRef v) const
{
	return(x != v.x || y != v.y || z != v.z);
}

/***************************************************************************/
// Non-equality
/***************************************************************************/
inline int Vector3::operator!=(
const ts_Vct3f &v) const
{
	return(x != v.X || y != v.Y || z != v.Z);
}

/***************************************************************************/
// Assignment
/***************************************************************************/
inline Vector3 &Vector3::operator=(
Vector3CRef v)
{
	x = v.x;
	y = v.y;
	z = v.z;
	return(*this);
}

/***************************************************************************/
/***************************************************************************/
inline Vector3 &Vector3::operator=(
const Vector4 &v)
{
	x = v.x;
	y = v.y;
	z = v.z;
	return(*this);
}

/***************************************************************************/
// Assignment
/***************************************************************************/
inline Vector3 &Vector3::operator=(
const float f)
{
	x = y = z = f;
	return(*this);
}

/***************************************************************************/
// Assignment
/***************************************************************************/
inline Vector3 &Vector3::operator=(
int i)
{
	x = y = z = (float)i;
	return(*this);
}

/***************************************************************************/
// Assignment
/***************************************************************************/
inline Vector3 &Vector3::operator=(
const ts_Vct3f &v)
{
	x = v.X;
	y = v.Y;
	z = v.Z;
	return(*this);
}

/***************************************************************************/
// Addition
/***************************************************************************/
inline Vector3 &Vector3::operator+=(
Vector3CRef v)
{
	x += v.x;
	y += v.y;
	z += v.z;

	return(*this);
}

/***************************************************************************/
// Addition
/***************************************************************************/
inline Vector3 &Vector3::operator+=(
const ts_Vct3f &v)
{
	x += v.X;
	y += v.Y;
	z += v.Z;

	return(*this);
}

/***************************************************************************/
// Subtraction
/***************************************************************************/
inline Vector3 &Vector3::operator-=(
Vector3CRef v)
{
	x -= v.x;
	y -= v.y;
	z -= v.z;

	return(*this);
}

/***************************************************************************/
// Subtraction
/***************************************************************************/
inline Vector3 &Vector3::operator-=(
const ts_Vct3f &v)
{
	x -= v.X;
	y -= v.Y;
	z -= v.Z;

	return(*this);
}

/***************************************************************************/
// Scaling
/***************************************************************************/
inline Vector3 &Vector3::operator*=(
float scale)
{
	x *= scale;
	y *= scale;
	z *= scale;

	return(*this);
}

/***************************************************************************/
// Scaling
/***************************************************************************/
inline Vector3 &Vector3::operator/=(
float scale)
{
	float invScale = 1.0f / scale;

	x *= invScale;
	y *= invScale;
	z *= invScale;

	return(*this);
}

/***************************************************************************/
// Addition
/***************************************************************************/
inline Vector3 Vector3::operator+(
Vector3CRef v) const
{
	Vector3 temp;
	temp.x = x + v.x;
	temp.y = y + v.y;
	temp.z = z + v.z;
	return(temp);
}

/***************************************************************************/
// Addition
/***************************************************************************/
inline Vector3 Vector3::operator+(
const ts_Vct3f &v) const
{
	Vector3 temp;
	temp.x = x + v.X;
	temp.y = y + v.Y;
	temp.z = z + v.Z;
	return(temp);
}

/***************************************************************************/
// Subtraction
/***************************************************************************/
inline Vector3 Vector3::operator-(
Vector3CRef v) const
{
	Vector3 temp;
	temp.x = x - v.x;
	temp.y = y - v.y;
	temp.z = z - v.z;
	return(temp);
}

/***************************************************************************/
// Subtraction
/***************************************************************************/
inline Vector3 Vector3::operator-(
const ts_Vct3f &v) const
{
	Vector3 temp;
	temp.x = x - v.X;
	temp.y = y - v.Y;
	temp.z = z - v.Z;
	return(temp);
}

/***************************************************************************/
// Scaling
/***************************************************************************/
inline Vector3 Vector3::operator*(
float scale) const
{
	Vector3 temp;
	temp.x = x * scale;
	temp.y = y * scale;
	temp.z = z * scale;
	return(temp);
}

/***************************************************************************/
// Scaling
/***************************************************************************/
inline Vector3 Vector3::operator/(
float scale) const
{
	float invScale = 1.0f / scale;
	Vector3 temp;
	temp.x = x * invScale;
	temp.y = y * invScale;
	temp.z = z * invScale;
	return(temp);
}

/***************************************************************************/
// Negation
/***************************************************************************/
inline Vector3 Vector3::operator-(
void) const
{
	Vector3 temp;
	temp.x = -x;
	temp.y = -y;
	temp.z = -z;
	return(temp);
}

/***************************************************************************/
// Get at the Vector3 like an array of reals
/***************************************************************************/
inline float &Vector3::operator[](
int x0y1z2)
{
	return(((float *)&x)[x0y1z2]);
}

/***************************************************************************/
// Casts
/***************************************************************************/
inline Vector3::operator ts_Vct3f*(void) const
{
	return((ts_Vct3f *)&x);
}

/***************************************************************************/
/***************************************************************************/
inline Vector3::operator ts_Vct3f(void) const
{
	return(*((ts_Vct3f *)&x));
}

/***************************************************************************/
/***************************************************************************/
inline Vector3::operator float*(void)
{
	return(&x);
}

/***************************************************************************/
// assignment constructor
/***************************************************************************/
inline Vector4::Vector4(
float x,
float y,
float z,
float w)
{
	this->x = x;
	this->y = y;
	this->z = z;
	this->w = w;
}

/***************************************************************************/
// Constructor for a Vector2
/***************************************************************************/
inline Vector4::Vector4(
const Vector2 &v)
{
	x = v.x;
	y = v.y;
	z = 0.0f;
	w = 1.0f;
}

/***************************************************************************/
// Constructor for a Vector3
/***************************************************************************/
inline Vector4::Vector4(
Vector3CRef v)
{
	x = v.x;
	y = v.y;
	z = v.z;
	w = 1.0f;
}

/***************************************************************************/
/***************************************************************************/
inline Vector4::Vector4(
const float *array)
{
	x = array[0];
	y = array[1];
	z = array[2];
	w = array[3];
}

/***************************************************************************/
/***************************************************************************/
inline Vector4::Vector4(
const ts_Vct2f &v)
{
	x = v.X;
	y = v.Y;
	z = 0.0f;
	w = 1.0f;
}

/***************************************************************************/
/***************************************************************************/
inline Vector4::Vector4(
const ts_Vct3f &v)
{
	x = v.X;
	y = v.Y;
	z = v.Z;
	w = 1.0f;
}

/***************************************************************************/
/***************************************************************************/
inline Vector4::Vector4(
const ts_Vct4f &v)
{
	x = v.X;
	y = v.Y;
	z = v.Z;
	w = v.W;
}

/***************************************************************************/
// Clear to 0
/***************************************************************************/
inline void Vector4::Clear(void)
{
	x = y = z = 0.0f;
	w = 1.0f;
}

/***************************************************************************/
// Set the values
/***************************************************************************/
inline void Vector4::Set(
float x,
float y,
float z,
float w)
{
	this->x = x;
	this->y = y;
	this->z = z;
	this->w = w;
}

/***************************************************************************/
// Set the values
/***************************************************************************/
inline void Vector4::Set(
const float *xyzwArray)
{
	x = xyzwArray[0];
	y = xyzwArray[1];
	z = xyzwArray[2];
	w = xyzwArray[3];
}

/***************************************************************************/
// Get at it
/***************************************************************************/
inline void Vector4::Get3(
ts_Vct3f &v) const
{
	v.X = x;
	v.Y = y;
	v.Z = z;
}

/***************************************************************************/
// Get at it
/***************************************************************************/
inline void Vector4::Get4(
ts_Vct4f &v) const
{
	v.X = x;
	v.Y = y;
	v.Z = z;
	v.W = w;
}

/***************************************************************************/
// Return this Dot v
/***************************************************************************/
inline float Vector4::Dot(
const Vector4 &v) const
{
	return(x * v.x + y * v.y + z * v.z);
}

/***************************************************************************/
// Return this X v
/***************************************************************************/
inline Vector4 Vector4::Cross(
const Vector4 &v) const
{
	Vector4 temp;
	temp.x = y * v.z - v.y * z;
	temp.y = -(x * v.z - v.x * z);
	temp.z = x * v.y - v.x * y;
	temp.w = 1.0f;
	return(temp);
}


/***************************************************************************/
// Return [this->x * v.x, this->y * v.y, this->z * v.z, this->w * v.w]
/***************************************************************************/
inline Vector4 Vector4::Diag(
const Vector4 &v) const
{
	Vector4 temp;
	temp.x = x * v.x;
	temp.y = y * v.y;
	temp.z = z * v.z;
	temp.w = w * v.w;
	return(temp);
}

/***************************************************************************/
// Return the Length of this Vector4
/***************************************************************************/
inline float Vector4::Length(void) const
{
	return(Math::Sqrt(x * x + y * y + z * z));
}

/***************************************************************************/
// Return the Length squared of this vector
/***************************************************************************/
inline float Vector4::LengthSquared(void) const
{
	return(x * x + y * y + z * z);
}

/***************************************************************************/
// Normalize the Vector4
/***************************************************************************/
inline void Vector4::Normalize(void)
{
	float invLength = 1.0f / Math::Sqrt(x * x + y * y + z * z);
	x *= invLength;
	y *= invLength;
	z *= invLength;
}

/***************************************************************************/
// Return the normalized Vector
/***************************************************************************/
inline Vector4 Vector4::Normalized(void) const
{
	float invLength = 1.0f / Math::Sqrt(x * x + y * y + z * z);
	Vector4 n;
	n.x = x * invLength;
	n.y = y * invLength;
	n.z = z * invLength;
	n.w = 1.0f;
	return(n);
}

/***************************************************************************/
// Negate the Vector4
/***************************************************************************/
inline void Vector4::Negate(void)
{
	x = -x;
	y = -y;
	z = -z;
}

/***************************************************************************/
// Return v1 Dot v2
/***************************************************************************/
inline float Vector4::Dot(
const Vector4 &v1,
const Vector4 &v2)
{
	return(v1.x * v2.x + v1.y * v2.y + v1.z * v2.z);
}

/***************************************************************************/
// Return v1 X v2
/***************************************************************************/
inline Vector4 Vector4::Cross(
const Vector4 &v1,
const Vector4 &v2)
{
	Vector4 temp;
	temp.x = v1.y * v2.z - v2.y * v1.z;
	temp.y = -(v1.x * v2.z - v2.x * v1.z);
	temp.z = v1.x * v2.y - v2.x * v1.y;
	temp.w = 1.0f;
	return(temp);
}

/***************************************************************************/
// Return [v1.x * v2.x, v1.y * v2.y, v1.z * v2.z, v1.w * v2.w]
/***************************************************************************/
inline Vector4 Vector4::Diag(
const Vector4 &v1,
const Vector4 &v2)
{
	Vector4 temp;
	temp.x = v1.x * v2.x;
	temp.y = v1.y * v2.y;
	temp.z = v1.z * v2.z;
	temp.w = v1.w * v2.w;
	return(temp);
}

/***************************************************************************/
// Return the Length of this Vector4
/***************************************************************************/
inline float Vector4::Length(
const Vector4 &v)
{
	return(Math::Sqrt(v.x * v.x + v.y * v.y + v.z * v.z));
}

/***************************************************************************/
// Normalize the Vector4
/***************************************************************************/
inline void Vector4::Normalize(
Vector4 &v)
{
	float invLength = 1.0f / Math::Sqrt(v.x * v.x + v.y * v.y + v.z * v.z);
	v.x *= invLength;
	v.y *= invLength;
	v.z *= invLength;
}

/***************************************************************************/
// Equality
/***************************************************************************/
inline int Vector4::operator==(
const Vector4 &v) const
{
	return(x == v.x && y == v.y && z == v.z && w == v.w);
}

/***************************************************************************/
// Non-equality
/***************************************************************************/
inline int Vector4::operator!=(
const Vector4 &v) const
{
	return(x != v.x || y != v.y || z != v.z || w != v.w);
}

/***************************************************************************/
// Assignment
/***************************************************************************/
inline Vector4 &Vector4::operator=(
const Vector4 &v)
{
	x = v.x;
	y = v.y;
	z = v.z;
	w = v.w;
	return(*this);
}

/***************************************************************************/
/***************************************************************************/
inline Vector4 &Vector4::operator=(
Vector3CRef v)
{
	x = v.x;
	y = v.y;
	z = v.z;
	w = 1.0f;
	return(*this);
}

/***************************************************************************/
// Assignment
/***************************************************************************/
inline Vector4 &Vector4::operator=(
const float f)
{
	x = y = z = f;
	w = 1.0f;
	return(*this);
}

/***************************************************************************/
// Assignment
/***************************************************************************/
inline Vector4 &Vector4::operator=(
int i)
{
	x = y = z = (float)i;
	w = 1.0f;
	return(*this);
}

/***************************************************************************/
// Addition
/***************************************************************************/
inline Vector4 &Vector4::operator+=(
const Vector4 &v)
{
	x += v.x;
	y += v.y;
	z += v.z;

	return(*this);
}

/***************************************************************************/
// Subtraction
/***************************************************************************/
inline Vector4 &Vector4::operator-=(
const Vector4 &v)
{
	x -= v.x;
	y -= v.y;
	z -= v.z;

	return(*this);
}

/***************************************************************************/
// Scaling
/***************************************************************************/
inline Vector4 &Vector4::operator*=(
float scale)
{
	x *= scale;
	y *= scale;
	z *= scale;

	return(*this);
}

/***************************************************************************/
// Scaling
/***************************************************************************/
inline Vector4 &Vector4::operator/=(
float scale)
{
	float invScale = 1.0f / scale;

	x *= invScale;
	y *= invScale;
	z *= invScale;

	return(*this);
}

/***************************************************************************/
// Addition
/***************************************************************************/
inline Vector4 Vector4::operator+(
const Vector4 &v) const
{
	Vector4 temp;
	temp.x = x + v.x;
	temp.y = y + v.y;
	temp.z = z + v.z;
	temp.w = 1.0f;
	return(temp);
}

/***************************************************************************/
// Subtraction
/***************************************************************************/
inline Vector4 Vector4::operator-(
const Vector4 &v) const
{
	Vector4 temp;
	temp.x = x - v.x;
	temp.y = y - v.y;
	temp.z = z - v.z;
	temp.w = 1.0f;
	return(temp);
}

/***************************************************************************/
// Scaling
/***************************************************************************/
inline Vector4 Vector4::operator*(
float scale) const
{
	Vector4 temp;
	temp.x = x * scale;
	temp.y = y * scale;
	temp.z = z * scale;
	temp.w = 1.0f;
	return(temp);
}

/***************************************************************************/
// Scaling
/***************************************************************************/
inline Vector4 Vector4::operator/(
float scale) const
{
	float invScale = 1.0f / scale;
	Vector4 temp;
	temp.x = x * invScale;
	temp.y = y * invScale;
	temp.z = z * invScale;
	temp.w = 1.0f;
	return(temp);
}

/***************************************************************************/
// Negation
/***************************************************************************/
inline Vector4 Vector4::operator-(
void) const
{
	Vector4 temp;
	temp.x = -x;
	temp.y = -y;
	temp.z = -z;
	temp.w = w;
	return(temp);
}

/***************************************************************************/
// Get at the Vector4 like an array of reals
/***************************************************************************/
inline float &Vector4::operator[](
int x0y1z2w3)
{
	return(((float *)&x)[x0y1z2w3]);
}

/***************************************************************************/
// Casts
/***************************************************************************/
inline Vector4::operator ts_Vct4f*(void) const
{
	return((ts_Vct4f *)&x);
}

/***************************************************************************/
/***************************************************************************/
inline Vector4::operator ts_Vct4f(void) const
{
	return(*((ts_Vct4f *)&x));
}

/***************************************************************************/
/***************************************************************************/
inline Vector4::operator float*(void)
{
	return(&x);
}

/***************************************************************************/
// Handy utilitiies
/***************************************************************************/
inline Vector2 operator*(float scalar, const Vector2 &v)
{
	return(v * scalar);
}

/***************************************************************************/
// Handy utilitiies
/***************************************************************************/
inline Vector3 operator*(
float scalar, 
Vector3CRef v)
{
	return(v * scalar);
}

/***************************************************************************/
// Handy utilitiies
/***************************************************************************/
inline Vector4 operator*(
float scalar, 
const Vector4 &v)
{
	return(v * scalar);
}

/***************************************************************************/
// handy utility (axis should be normalized)
/***************************************************************************/
inline Vector3 RotatePointAroundAxis(
Vector3CRef p,
Vector3CRef axis,
float theta)
{
	float cTheta = Math::Cos(theta);
	float sTheta = Math::Sin(theta);

	return(cTheta * p +
			 (1.0f - cTheta) * axis * Vector3::Dot(axis, p) +
			 sTheta * Vector3::Cross(axis, p));
}

// Miscellaneous stuff.
inline float Min(float a, float b)
{
	if (a < b)
		return a;
	return b;
}

inline float Max(float a, float b)
{
	if (a < b)
		return b;
	return a;
}


//------------------------------------------------------------------------
// NearestPointOnLineSeg
//    Utility to compute the nearest point on a line segment to a point.
//    Two versions are provided each for Vector2 and Vector3.
//    This technique calculates this by doing an orthogonal projection
//    of the point on the line, then making sure the point is on the line
//    segment.  If it's not then the closest endpoint of the line segment
//    is used.
//------------------------------------------------------------------------
//---------------------------------------------
//  2D Version
//---------------------------------------------
inline void NearestPointOnLineSeg(const Vector2& p, const Vector2& l1, const Vector2& l2,
								  Vector2& out)
{
	//---------------------------------------------
	//  ..........IMPORTANT NOTE..........
	//  If you make changes to this code you must
	//  seriously consider making the same changes
	//  to the 3D version of this routine.
	//---------------------------------------------
	Vector2 A = l2-l1;
	float dot = A.Dot(p-l1);

	// If the dot product is 0 or less than one, then the vectors are pointing
	// away from each other and we can use (l1);
	if (dot <= .0001f)
	{
		out = l1;
		return;
	}

	out = l1 + A * dot / A.LengthSquared();

	// Do a bounding rectangle check to see if the point is actually on the line.
	if (out.x < Min(l1.x, l2.x) || out.x > Max(l1.x, l2.x) ||
		out.y < Min(l1.y, l2.y) || out.y > Max(l1.y, l2.y))
	{
		// We know the point isn't on the line, and we know that the vectors
		// aren't opposite direction.
		out = l2;
	}
}

//---------------------------------------------
//  3D Version
//---------------------------------------------
inline void NearestPointOnLineSeg(const Vector3& p, const Vector3& l1, const Vector3& l2,
								  Vector3& out)
{
	//---------------------------------------------
	//  ..........IMPORTANT NOTE..........
	//  If you make changes to this code you must
	//  seriously consider making the same changes
	//  to the 2D version of this routine.
	//---------------------------------------------
	Vector3 A = l2-l1;
	float dot = A.Dot(p-l1);

	// If the dot product is 0 or less than one, then the vectors are pointing
	// away from each other and we can use (l1);
	if (dot <= .0001f)
	{
		out = l1;
		return;
	}

	out = l1 + A * dot / A.LengthSquared();

	// Do a bounding rectangle check to see if the point is actually on the line.
	if (out.x < Min(l1.x, l2.x) || out.x > Max(l1.x, l2.x) ||
		out.y < Min(l1.y, l2.y) || out.y > Max(l1.y, l2.y) ||
		out.z < Min(l1.z, l2.z) || out.z > Max(l1.z, l2.z))
	{
		// We know the point isn't on the line, and we know that the vectors
		// aren't opposite direction.
		out = l2;
	}
}

//----------------------------------------------------------------------
// NearestPointOnLine
//    Find the closest point on an unbounded line by doing an
//    orthogonal projection of the point on the line.
//----------------------------------------------------------------------
//---------------------------------------------
//  2D Version
//---------------------------------------------
inline void NearestPointOnLine(const Vector2& p, const Vector2& l1, const Vector2& l2,
							   Vector2& out)
{
	//---------------------------------------------
	//  ..........IMPORTANT NOTE..........
	//  If you make changes to this code you must
	//  seriously consider making the same changes
	//  to the 3D version of this routine.
	//---------------------------------------------
	Vector2 A = l2 - l1;
	out = l1 + A.Dot(p-l1) * A / A.LengthSquared();
}

//---------------------------------------------
//  3D Version
//---------------------------------------------
inline void NearestPointOnLine(const Vector3& p, const Vector3& l1, const Vector3& l2,
							   Vector3& out)
{
	//---------------------------------------------
	//  ..........IMPORTANT NOTE..........
	//  If you make changes to this code you must
	//  seriously consider making the same changes
	//  to the 2D version of this routine.
	//---------------------------------------------
	Vector3 A = l2 - l1;
	out = l1 + A.Dot(p-l1) * A / A.LengthSquared();
}


 //---------------------------------------------
 //  Convenience functions
 //---------------------------------------------
inline Vector2 NearestPointOnLineSeg(const Vector2& p, const Vector2& l1, const Vector2& l2)
{
	Vector2 out;
	NearestPointOnLineSeg(p, l1, l2, out);
	return out;
}

inline Vector3 NearestPointOnLineSeg(const Vector3& p, const Vector3& l1, const Vector3& l2)
{
	Vector3 out;
	NearestPointOnLineSeg(p, l1, l2, out);
	return out;
}

inline Vector2 NearestPointOnLine(const Vector2& p, const Vector2& l1, const Vector2& l2)
{
	Vector2 out;
	NearestPointOnLine(p, l1, l2, out);
	return out;
}

inline Vector3 NearestPointOnLine(const Vector3& p, const Vector3& l1, const Vector3& l2)
{
	Vector3 out;
	NearestPointOnLine(p, l1, l2, out);
	return out;
}

// Simple thing for comparing points in space
inline float Distance(Vector3 &a, Vector3 &b)
{
	return((a - b).Length());
}

// Simple thing for comparing points in space
inline float DistanceSquared(Vector3 &a, Vector3 &b)
{
	return((a - b).LengthSquared());
}

// Simple thing for comparing points in space
inline bool PointsCoincide(Vector3 &a, Vector3 &b, float thresholdSquared)
{
	return((a - b).LengthSquared() < thresholdSquared);
}
