#include "TTLPCH.h"
#include "GameSettingConfig.hpp"

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Initialize default value and do a sanity check on them
///
/// \param pName Name of the configuration value
///
/// \param any Should we use the any value?
///
/// \param max Maximum value in the range
///
/// \param min Minimum value in the range
///
/// \param value Default value to set
///////////////////////////////////////////////////////////////////////////////////////////////////
IntegerType::IntegerType(const char *pName, int increment, int max, int min, int value)
    : ParamType(pName),
      _value(value),
      _min(min),
      _max(max),
      _increment(increment)
{
    // Ensure that parameters are sane
    ASSERT(min < max);
    ASSERT(value >= min);
    ASSERT(value <= max);
    ASSERT(_increment < (min + max));
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Get the string value for the current value.  If ANY, returns ANY, otherwise returns the
/// current list value.
///
/// \param A string for the current value.
///////////////////////////////////////////////////////////////////////////////////////////////////
/* virtual */ const char *IntegerType::getStringValue()
{
    sprintf(_stringValue, "%d", _value);
    return _stringValue;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Get the integer value for the current value.  If ANY, returns ANY, otherwise returns the
/// offset into the list.
///
/// \param An integer for the current value.
///////////////////////////////////////////////////////////////////////////////////////////////////
/* virtual */ int IntegerType::getIntegerValue()
{
    return _value;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Set the current value and sanity check it
///
/// \param value The new value to set
///////////////////////////////////////////////////////////////////////////////////////////////////
void IntegerType::setValue(int value)
{
    // Make sure the value is between min and max
    _value = min(max(value, _min), _max);
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Set the new minimum and sanity check it
///
/// \param min The new minimum value
///////////////////////////////////////////////////////////////////////////////////////////////////
void IntegerType::setMin(int min)
{
    ASSERT(min < _max);

    _min = min;
    _value = max(_value, _min);
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Set the new maximum and sanity check it
///
/// \param max The new maximum value
///////////////////////////////////////////////////////////////////////////////////////////////////
void IntegerType::setMax(int max)
{
    ASSERT(_min < max);

    _max = max;
    _value = min(_value, _max);
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Go to the next value (including ANY if it's enabled)
///////////////////////////////////////////////////////////////////////////////////////////////////
/* virtual */ void IntegerType::nextValue()
{
    _value += _increment;

	if (_value > _max)
    {
        _value = _min;
    }
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Go to the previous value (including ANY if it's enabled)
///////////////////////////////////////////////////////////////////////////////////////////////////
/* virtual */ void IntegerType::prevValue()
{
    _value -= _increment;

    if (_value < _min)
    {
        _value = _max;
    }
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Initialize the name and any value
///
/// \param pName Name of the configuration value
///
/// \param any Should we use the any value?
///////////////////////////////////////////////////////////////////////////////////////////////////
StringType::StringType(const char *pName)
    : ParamType(pName)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Get the string value for the current value.  If ANY, returns ANY, otherwise returns the
/// current list value.
///
/// \param A string for the current value.
///////////////////////////////////////////////////////////////////////////////////////////////////
/* virtual */ const char *StringType::getStringValue()
{
    return _values.getCurrent();
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Get the integer value for the current value.  If ANY, returns ANY, otherwise returns the
/// offset into the list.
///
/// \param An integer for the current value.
///////////////////////////////////////////////////////////////////////////////////////////////////
/* virtual */ int StringType::getIntegerValue()
{
    return _values.getPosition();
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Get the description for the current value.  If ANY, returns ANY.
///////////////////////////////////////////////////////////////////////////////////////////////////
/* virtual */ const char* StringType::getDescription()
{
	return _descriptions.getCurrent();
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Go to the next value (including ANY if it's enabled)
///////////////////////////////////////////////////////////////////////////////////////////////////
/* virtual */ void StringType::nextValue()
{
    // If we're not using ANY wrap around
    if (_values.getCurrent() == _values.getBack())
    {
        _values.begin();
		_descriptions.begin();
    }
    // Otherwise, just move to the next element
    else
    {
        _values.next();
		_descriptions.next();
    }
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Go to the previous value (including ANY if it's enabled)
///////////////////////////////////////////////////////////////////////////////////////////////////
/* virtual */ void StringType::prevValue()
{
    if (_values.getCurrent() == _values.getFront())
    {
        _values.end();
		_descriptions.end();
    }
    // Otherwise, just move to the next element
    else
    {
        _values.previous();
		_descriptions.previous();
    }
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Add a new string to the type
///
/// \param pValue The value of the new string to add
///////////////////////////////////////////////////////////////////////////////////////////////////
void StringType::addStringValue(const char *pValue, const char *pDescription)
{
    _values.push_back(pValue);
	if(pDescription == NULL)
	{
		_descriptions.push_back(" ");
	}
	else
	{
		_descriptions.push_back(pDescription);
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Find a value's index in the container, or -1
///
/// \param pValue The value of the string to find
///////////////////////////////////////////////////////////////////////////////////////////////////
int StringType::getIndexForValue( const char* value )
{
	_values.begin();
	for( unsigned int i = 0; i < _values.count(); i++ )
	{
		if( !strcmp( _values.getCurrent(), value ) )
		{
			return i;
		}
		_values.next();
	}
	return -1;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Get the total number of values in the type
///
/// \return The total number of values in the type
///////////////////////////////////////////////////////////////////////////////////////////////////
int StringType::getNumValues()
{
    return _values.count();
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Initialize the name and any value
///
/// \param pName Name of the configuration value
///
/// \param any Should we use the any value?
///////////////////////////////////////////////////////////////////////////////////////////////////
EnumType::EnumType(const char *pName)
    : ParamType(pName)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Get the string value for the current value.  If ANY, returns ANY, otherwise returns the
/// current list value.
///
/// \param A string for the current value.
///////////////////////////////////////////////////////////////////////////////////////////////////
/* virtual */ const char *EnumType::getStringValue()
{
    return _values.getCurrent()->stringValue;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Get the integer value for the current value.  If ANY, returns ANY, otherwise returns the
/// offset into the list.
///////////////////////////////////////////////////////////////////////////////////////////////////
/* virtual */ int EnumType::getIntegerValue()
{
    return _values.getCurrent()->intValue;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Get the description for the current value.  If ANY, returns ANY.
///////////////////////////////////////////////////////////////////////////////////////////////////
/* virtual */ const char* EnumType::getDescription()
{
	return _descriptions.getCurrent();
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Go to the next value (including ANY if it's enabled)
///////////////////////////////////////////////////////////////////////////////////////////////////
/* virtual */ void EnumType::nextValue()
{
    if (_values.getCurrent() == _values.getBack())
    {
        _values.begin();
		_descriptions.begin();
    }
    // Otherwise, just move to the next element
    else
    {
        _values.next();
		_descriptions.next();
    }
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Go to the previous value (including ANY if it's enabled)
///////////////////////////////////////////////////////////////////////////////////////////////////
/* virtual */ void EnumType::prevValue()
{
    if (_values.getCurrent() == _values.getFront())
    {
        _values.end();
		_descriptions.end();
    }
    // Otherwise, just move to the next element
    else
    {
        _values.previous();
		_descriptions.previous();
    }
}

///////////////////////////////////////////////////////////////////////////////////////////////////
/// Add a new value to the type
///
/// \param integer The integer portion of the new value to add
///
/// \param pString The string portion of the new value to add
///////////////////////////////////////////////////////////////////////////////////////////////////
void EnumType::addValue(int integer, const char *pString, const char *pDescription)
{
    Value newValue;

    newValue.intValue = integer;
    strncpy(newValue.stringValue, pString, 50);

    _values.push_back(newValue);

	if(pDescription == NULL)
	{
		_descriptions.push_back(" ");
	}
	else
	{
		_descriptions.push_back(pDescription);
	}
}
